<?php
/**
 * Plugin Name: WP InView
 * Description: Lightweight animation plugin that adds modern scroll and page transition effects to WordPress blocks. Configure animations directly from the block sidebar, with precise control over triggers, timing, easing and global settings.
 * Version: 0.9.0
 * Author: WP InView (wpinview.com)
 * Text Domain: wp-inview
 * Requires at least: 6.0
 * Requires PHP: 7.4
 */

if (! defined('ABSPATH')) {
	exit;
}

define('WP_INVIEW_MIN_PHP', '7.4');
define('WP_INVIEW_MIN_WP', '6.0');

/**
 * Quick link "Ustawienia" in Plugins list.
 * Target: /wp-admin/options-general.php?page=wp-inview
 */
add_filter('plugin_action_links_wp-inview/wp-inview.php', function (array $links): array {
	$url = admin_url('options-general.php?page=wp-inview');
	array_unshift(
		$links,
		sprintf('<a href="%s">%s</a>', esc_url($url), esc_html__('Settings', 'wp-inview'))
	);
	return $links;
});
/**
 * Row meta links under plugin description in Plugins list.
 * Adds: "Zobacz szczegóły" -> docs URL
 */
add_filter('plugin_row_meta', function (array $links, string $file): array {
	if ($file !== 'wp-inview/wp-inview.php') {
		return $links;
	}

	$docs_url = 'https://example.com/docs/wp-inview'; // podmień później

	$links[] = sprintf(
		'<a href="%s" target="_blank" rel="noopener">%s</a>',
		esc_url($docs_url),
		esc_html__('View details', 'wp-inview')
	);

	return $links;
}, 10, 2);
/**
 * Admin notice helper.
 */
function wp_inview_admin_notice_error(string $message): void {
	add_action('admin_notices', function () use ($message) {
		echo '<div class="notice notice-error"><p>' . esc_html($message) . '</p></div>';
	});
}

/**
 * Deactivate self (only in admin).
 */
function wp_inview_deactivate_self(): void {
	if (is_admin() && function_exists('deactivate_plugins')) {
		deactivate_plugins(plugin_basename(__FILE__));
	}
}

/**
 * 1) PHP version check BEFORE loading any strict_types files.
 */
if (version_compare(PHP_VERSION, WP_INVIEW_MIN_PHP, '<')) {
	wp_inview_admin_notice_error(sprintf('WP InView requires PHP %s or higher.', WP_INVIEW_MIN_PHP));
	add_action('admin_init', 'wp_inview_deactivate_self');
	return;
}

/**
 * 2) WP version check.
 */
global $wp_version;
$cur_wp = is_string($wp_version ?? null) ? $wp_version : (string) get_bloginfo('version');

if (version_compare($cur_wp, WP_INVIEW_MIN_WP, '<')) {
	wp_inview_admin_notice_error(sprintf('WP InView requires WordPress %s or higher.', WP_INVIEW_MIN_WP));
	add_action('admin_init', 'wp_inview_deactivate_self');
	return;
}

/**
 * Core plugin constants.
 * Keep aliases for backward-compatibility with current codebase.
 */
if (! defined('WP_INVIEW_VERSION')) {
	define('WP_INVIEW_VERSION', '0.9.0');
}

if (! defined('WP_INVIEW_FILE')) {
	define('WP_INVIEW_FILE', __FILE__);
}

if (! defined('WP_INVIEW_DIR')) {
	define('WP_INVIEW_DIR', plugin_dir_path(__FILE__));
}

if (! defined('WP_INVIEW_URL')) {
	define('WP_INVIEW_URL', plugin_dir_url(__FILE__));
}

if (! defined('WP_INVIEW_BASENAME')) {
	define('WP_INVIEW_BASENAME', plugin_basename(__FILE__));
}

/**
 * Legacy aliases used across the codebase.
 * Do not remove until all references are migrated.
 */
//if (! defined('WP_INVIEW_PATH')) {
//	define('WP_INVIEW_PATH', WP_INVIEW_DIR);
//}

//if (! defined('WP_INVIEW_BASE_URL')) {
//	define('WP_INVIEW_BASE_URL', WP_INVIEW_URL);
//}

$core_files = [
	'includes/Plugin.php',
	'includes/Loader.php',
];

$missing_files = [];
foreach ($core_files as $file) {
	$full_path = WP_INVIEW_DIR . $file;
	if (! file_exists($full_path)) {
		$missing_files[] = $file;
	}
}
function wp_inview_should_log(): bool {
	return (defined('WP_DEBUG') && WP_DEBUG)
		|| (defined('WP_INVIEW_DEBUG') && WP_INVIEW_DEBUG);
}

if (! empty($missing_files)) {
	add_action('admin_notices', function () use ($missing_files) {
		echo '<div class="notice notice-error"><p>';
		echo '<strong>WP InView Critical Error:</strong> ';
		echo esc_html__('Plugin files are missing or corrupted. Please reinstall the plugin.', 'wp-inview');
		echo '<br><small>' . esc_html__('Missing files:', 'wp-inview') . ' ';
		echo esc_html(implode(', ', $missing_files));
		echo '</small></p></div>';
	});

	if (wp_inview_should_log()) {
		error_log('[WP InView] CRITICAL: Missing core files: ' . implode(', ', $missing_files));
	}
	add_action('admin_init', 'wp_inview_deactivate_self');
	return;
}

// Now safe to require strict_types code
require_once WP_INVIEW_DIR . 'includes/Plugin.php';
require_once WP_INVIEW_DIR . 'includes/Loader.php';

try {
	WP_InView_Loader::load();
	WP_InView_Plugin::init();
} catch (\Throwable $e) {
	if (wp_inview_should_log()) {
		error_log('[WP InView] Plugin initialization failed: ' . $e->getMessage());
	}

	add_action('admin_notices', function () use ($e) {
		echo '<div class="notice notice-error"><p>';
		echo '<strong>WP InView Error:</strong> ';
		echo esc_html__('Plugin failed to initialize. Error: ', 'wp-inview');
		echo esc_html($e->getMessage());
		echo '</p></div>';
	});

	add_action('admin_init', 'wp_inview_deactivate_self');
}