<?php

declare (strict_types = 1);

if (! defined('ABSPATH')) {
	exit;
}

final class WP_InView_ViewHelpers {

	private const CLASS_MARKER = 'wp-inview';

	/**
	 * Single source of truth for mask directions across globals and block overrides.
	 *
	 * @var array<int,string>
	 */
	private const MASK_DIRECTIONS = ['up', 'down', 'left', 'right', 'horizontal', 'vertical'];

	public static function register(): void {
		// Very late, so other render_block filters don't overwrite our injected attrs.
		add_filter('render_block', [self::class, 'inject_animation'], 9999, 2);
	}

	public static function inject_animation(string $content, array $block): string {
		$attrs = isset($block['attrs']) && is_array($block['attrs']) ? $block['attrs'] : [];
		$cfg   = $attrs['wpInView'] ?? null;

		if (! is_array($cfg) || empty($cfg['enabled'])) {
			return $content;
		}

		$registry = WP_InView_Schema::presets_registry();

		$preset_raw = isset($cfg['preset']) ? (string) $cfg['preset'] : 'move';
		$preset     = self::sanitize_key($preset_raw);

		if (! isset($registry[$preset])) {
			if ($preset !== 'move') {
				WP_InView_Debug::warn('Unknown preset requested, falling back to move', [
					'preset' => $preset_raw,
				]);
			}
			$preset = 'move';
		}

		$use_global = ! isset($cfg['useGlobal']) || $cfg['useGlobal'] !== false;

		// Base always present.
		$data = [
			'data-wp-inview-preset' => $preset,
		];

		// Once can be used even with globals.
		if (array_key_exists('once', $cfg)) {
			$data['data-wp-inview-once'] = self::sanitize_bool($cfg['once'], true) ? '1' : '0';
		}

		// Cascade toggle only. Params remain global.
		if (! empty($cfg['cascade'])) {
			$data['data-wp-inview-cascade'] = '1';
		}

		// UI schema is the single source for ranges/allowed/maxLen.
		$ui        = WP_InView_Schema::ui_schema();
		$anim_ui   = is_array($ui['animation'] ?? null) ? $ui['animation'] : [];
		$fields_ui = is_array($anim_ui['fields'] ?? null) ? $anim_ui['fields'] : [];

		$easing_maxlen = self::ui_maxlen($fields_ui, 'easing', 120);

		/**
		 * Apply parameters either from:
		 * - block overrides ($cfg) when useGlobal = false
		 * - global settings (animation common + preset) when useGlobal = true
		 */
		if ($use_global) {
			$global_common = self::get_global_common();
			$global_preset = self::get_global_preset($preset);

			// Common fields.
			self::apply_float($data, $global_common, $fields_ui, 'opacityFrom', 'data-wp-inview-opacity-from', 0.0);
			self::apply_int($data, $global_common, $fields_ui, 'duration', 'data-wp-inview-duration', 650);
			self::apply_int($data, $global_common, $fields_ui, 'delay', 'data-wp-inview-delay', 0);
			self::apply_float($data, $global_common, $fields_ui, 'threshold', 'data-wp-inview-threshold', 0.15);

			if (array_key_exists('easing', $global_common)) {
				$easing = self::sanitize_easing($global_common['easing'], $easing_maxlen);
				if ($easing !== '') {
					$data['data-wp-inview-easing'] = $easing;
				}
			}

			if (array_key_exists('rootMargin', $global_common)) {
				$rm = self::sanitize_rootmargin($global_common['rootMargin']);
				if ($rm !== '') {
					$data['data-wp-inview-rootmargin'] = $rm;
				}
			}

			// Preset-specific fields from global preset settings.
			if ($preset === 'move') {
				if (array_key_exists('direction', $global_preset)) {
					$dir     = self::sanitize_key((string) $global_preset['direction']);
					$allowed = self::ui_allowed($fields_ui, 'direction', ['up', 'down', 'left', 'right']);
					if (in_array($dir, $allowed, true)) {
						$data['data-wp-inview-direction'] = $dir;
					}
				}

				self::apply_int($data, $global_preset, $fields_ui, 'distance', 'data-wp-inview-distance', 24);
			}

			if ($preset === 'zoom') {
				$allowed_modes = self::ui_allowed($fields_ui, 'zoomMode', ['in', 'out']);
				$mode          = array_key_exists('zoomMode', $global_preset) ? self::sanitize_key((string) $global_preset['zoomMode']) : 'in';
				if (! in_array($mode, $allowed_modes, true)) {
					$mode = 'in';
				}
				$data['data-wp-inview-zoom-mode'] = $mode;

				if (array_key_exists('scaleFrom', $global_preset)) {
					$range                             = self::ui_scale_range($fields_ui, $mode);
					$val                               = self::clamp_float($global_preset['scaleFrom'], $range['min'], $range['max'], ($mode === 'in') ? 0.9 : 1.15);
					$data['data-wp-inview-scale-from'] = (string) $val;
				}

				// rubberMode (new, preferred) — off|soft|medium|hard|custom
				if (array_key_exists('rubberMode', $global_preset)) {
					$allowed_rubber = self::ui_allowed($fields_ui, 'rubberMode', ['off', 'soft', 'medium', 'hard', 'custom']);
					$rubber_mode    = self::sanitize_key((string) $global_preset['rubberMode']);
					if (in_array($rubber_mode, $allowed_rubber, true)) {
						$data['data-wp-inview-rubber-mode'] = $rubber_mode;
						// Sync legacy bool attr — runtime uses it for CSS class
						$data['data-wp-inview-rubber'] = ($rubber_mode !== 'off') ? '1' : '0';
					}
				} elseif (array_key_exists('rubber', $global_preset)) {
					// Legacy bool fallback
					$rubber_bool                        = self::sanitize_bool($global_preset['rubber'], false);
					$data['data-wp-inview-rubber']      = $rubber_bool ? '1' : '0';
					$data['data-wp-inview-rubber-mode'] = $rubber_bool ? 'soft' : 'off';
				}

				// rubberAmp — only meaningful when rubberMode === 'custom'
				if (array_key_exists('rubberAmp', $global_preset)) {
					$amp                               = self::clamp_float($global_preset['rubberAmp'], 0, 0.1, 0.035);
					$data['data-wp-inview-rubber-amp'] = (string) $amp;
				}
			}

			if ($preset === 'blur') {
				self::apply_int($data, $global_preset, $fields_ui, 'blurFrom', 'data-wp-inview-blur-from', 25);
			}

			if ($preset === 'mask') {
				if (array_key_exists('maskVariant', $global_preset)) {
					$allowed = self::ui_allowed($fields_ui, 'maskVariant', ['wipe']);
					$v       = self::sanitize_key((string) $global_preset['maskVariant']);
					if (in_array($v, $allowed, true)) {
						$data['data-wp-inview-mask-variant'] = $v;
					}
				}

				if (array_key_exists('maskDirection', $global_preset)) {
					$allowed = self::ui_allowed($fields_ui, 'maskDirection', self::MASK_DIRECTIONS);
					$v       = self::sanitize_key((string) $global_preset['maskDirection']);
					if (in_array($v, $allowed, true)) {
						$data['data-wp-inview-mask-direction'] = $v;
					}
				}

				self::apply_int($data, $global_preset, $fields_ui, 'maskFeather', 'data-wp-inview-mask-feather', 24);
				self::apply_int($data, $global_preset, $fields_ui, 'maskInset', 'data-wp-inview-mask-inset', 20);
			}
		} else {
			// Block-level overrides.

			self::apply_float($data, $cfg, $fields_ui, 'opacityFrom', 'data-wp-inview-opacity-from', 0.0);
			self::apply_int($data, $cfg, $fields_ui, 'duration', 'data-wp-inview-duration', 650);
			self::apply_int($data, $cfg, $fields_ui, 'delay', 'data-wp-inview-delay', 0);
			self::apply_float($data, $cfg, $fields_ui, 'threshold', 'data-wp-inview-threshold', 0.15);

			if (array_key_exists('easing', $cfg)) {
				$easing = self::sanitize_easing($cfg['easing'], $easing_maxlen);
				if ($easing !== '') {
					$data['data-wp-inview-easing'] = $easing;
				}
			}

			if (array_key_exists('rootMargin', $cfg)) {
				// Unified validation across branches.
				$rm = self::sanitize_rootmargin($cfg['rootMargin']);
				if ($rm !== '') {
					$data['data-wp-inview-rootmargin'] = $rm;
				}
			}

			// Preset-specific fields.
			if ($preset === 'move') {
				if (array_key_exists('direction', $cfg)) {
					$dir     = self::sanitize_key((string) $cfg['direction']);
					$allowed = self::ui_allowed($fields_ui, 'direction', ['up', 'down', 'left', 'right']);
					if (in_array($dir, $allowed, true)) {
						$data['data-wp-inview-direction'] = $dir;
					}
				}

				self::apply_int($data, $cfg, $fields_ui, 'distance', 'data-wp-inview-distance', 24);
			}

			if ($preset === 'zoom') {
				$allowed_modes = self::ui_allowed($fields_ui, 'zoomMode', ['in', 'out']);
				$mode          = array_key_exists('zoomMode', $cfg) ? self::sanitize_key((string) $cfg['zoomMode']) : 'in';
				if (! in_array($mode, $allowed_modes, true)) {
					$mode = 'in';
				}
				$data['data-wp-inview-zoom-mode'] = $mode;

				if (array_key_exists('scaleFrom', $cfg)) {
					$range                             = self::ui_scale_range($fields_ui, $mode);
					$val                               = self::clamp_float($cfg['scaleFrom'], $range['min'], $range['max'], ($mode === 'in') ? 0.9 : 1.15);
					$data['data-wp-inview-scale-from'] = (string) $val;
				}

				// rubberMode (new, preferred) — off|soft|medium|hard|custom
				if (array_key_exists('rubberMode', $cfg)) {
					$allowed_rubber = self::ui_allowed($fields_ui, 'rubberMode', ['off', 'soft', 'medium', 'hard', 'custom']);
					$rubber_mode    = self::sanitize_key((string) $cfg['rubberMode']);
					if (in_array($rubber_mode, $allowed_rubber, true)) {
						$data['data-wp-inview-rubber-mode'] = $rubber_mode;
						// Sync legacy bool attr — runtime uses it for CSS class
						$data['data-wp-inview-rubber'] = ($rubber_mode !== 'off') ? '1' : '0';
					}
				} elseif (array_key_exists('rubber', $cfg)) {
					// Legacy bool fallback
					$rubber_bool                        = self::sanitize_bool($cfg['rubber'], false);
					$data['data-wp-inview-rubber']      = $rubber_bool ? '1' : '0';
					$data['data-wp-inview-rubber-mode'] = $rubber_bool ? 'soft' : 'off';
				}

				// rubberAmp — only meaningful when rubberMode === 'custom'
				if (array_key_exists('rubberAmp', $cfg)) {
					$amp                               = self::clamp_float($cfg['rubberAmp'], 0, 0.1, 0.035);
					$data['data-wp-inview-rubber-amp'] = (string) $amp;
				}
			}

			if ($preset === 'blur') {
				self::apply_int($data, $cfg, $fields_ui, 'blurFrom', 'data-wp-inview-blur-from', 25);
			}

			if ($preset === 'mask') {
				if (array_key_exists('maskVariant', $cfg)) {
					$allowed = self::ui_allowed($fields_ui, 'maskVariant', ['wipe']);
					$v       = self::sanitize_key((string) $cfg['maskVariant']);
					if (in_array($v, $allowed, true)) {
						$data['data-wp-inview-mask-variant'] = $v;
					}
				}

				if (array_key_exists('maskDirection', $cfg)) {
					$allowed = self::ui_allowed($fields_ui, 'maskDirection', self::MASK_DIRECTIONS);
					$v       = self::sanitize_key((string) $cfg['maskDirection']);
					if (in_array($v, $allowed, true)) {
						$data['data-wp-inview-mask-direction'] = $v;
					}
				}

				self::apply_int($data, $cfg, $fields_ui, 'maskFeather', 'data-wp-inview-mask-feather', 24);
				self::apply_int($data, $cfg, $fields_ui, 'maskInset', 'data-wp-inview-mask-inset', 20);
			}
		}

		// If we cannot parse HTML safely, do nothing.
		if (! class_exists('WP_HTML_Tag_Processor')) {
			WP_InView_Debug::warn('WP_HTML_Tag_Processor not available, cannot inject attrs');
			return $content;
		}

		// Signal to Assets that at least one animated block exists on this page.
		if (class_exists('WP_InView_Assets') && method_exists('WP_InView_Assets', 'mark_frontend_needed')) {
			WP_InView_Assets::mark_frontend_needed();
		}

		$p = new WP_HTML_Tag_Processor($content);

		// Find first meaningful tag to annotate.
		while ($p->next_tag()) {
			$tag = strtoupper((string) $p->get_tag());
			if ($tag === 'SCRIPT' || $tag === 'STYLE' || $tag === 'NOSCRIPT') {
				continue;
			}

			$existing = (string) $p->get_attribute('class');
			$classes  = preg_split('/\s+/', trim($existing)) ?: [];

			if (! in_array(self::CLASS_MARKER, $classes, true)) {
				$classes[] = self::CLASS_MARKER;
			}

			if (! in_array('wp-inview--ready', $classes, true)) {
				$classes[] = 'wp-inview--ready';
			}

			$presetClass = 'wp-inview--preset-' . $preset;
			if (! in_array($presetClass, $classes, true)) {
				$classes[] = $presetClass;
			}

			// Important: never add wp-inview--inview here. If it exists in HTML, remove it.
			$classes = array_values(array_diff($classes, ['wp-inview--inview']));

			if ($preset === 'blur' && ! in_array('wp-inview--preset-blur', $classes, true)) {
				$classes[] = 'wp-inview--preset-blur';
			}

			$p->set_attribute('class', trim(implode(' ', array_filter($classes))));

			foreach ($data as $k => $v) {
				$p->set_attribute($k, $v);
			}

			// For blur: set CSS var inline so above-the-fold elements start blurred.
			if ($preset === 'blur' && isset($data['data-wp-inview-blur-from'])) {
				$blur  = (int) $data['data-wp-inview-blur-from'];
				$style = (string) $p->get_attribute('style');

				$style = trim($style);
				if ($style !== '' && substr($style, -1) !== ';') {
					$style .= ';';
				}
				$style .= ' --wp-inview-blur-from: ' . $blur . 'px;';

				$p->set_attribute('style', trim($style));
			}

			// Ensure mask attrs are not leaked into other presets.
			if ($preset !== 'mask') {
				$p->remove_attribute('data-wp-inview-mask-variant');
				$p->remove_attribute('data-wp-inview-mask-direction');
				$p->remove_attribute('data-wp-inview-mask-feather');
				$p->remove_attribute('data-wp-inview-mask-inset');
			}

			return $p->get_updated_html();
		}

		WP_InView_Debug::warn('No suitable tag found to inject wp-inview attributes', [
			'blockName' => isset($block['blockName']) ? (string) $block['blockName'] : '',
		]);

		return $content;
	}

	// Global settings helpers

	/**
	 * Global common animation settings (without presets subtree).
	 *
	 * @return array<string,mixed>
	 */
	private static function get_global_common(): array {
		if (! class_exists('WP_InView_Settings') || ! method_exists('WP_InView_Settings', 'get')) {
			return [];
		}

		$s = (array) WP_InView_Settings::get();
		$a = isset($s['animation']) && is_array($s['animation']) ? $s['animation'] : [];

		// Remove presets, leave only common animation fields.
		if (isset($a['presets'])) {
			unset($a['presets']);
		}

		return $a;
	}

	/**
	 * Global preset settings for a given preset key.
	 *
	 * @return array<string,mixed>
	 */
	private static function get_global_preset(string $preset): array {
		if (! class_exists('WP_InView_Settings') || ! method_exists('WP_InView_Settings', 'get')) {
			return [];
		}

		$s = (array) WP_InView_Settings::get();
		$a = isset($s['animation']) && is_array($s['animation']) ? $s['animation'] : [];
		$p = isset($a['presets']) && is_array($a['presets']) ? $a['presets'] : [];

		$gp = $p[$preset] ?? [];
		return is_array($gp) ? $gp : [];
	}

	// UI helpers

	/**
	 * @param array<string,mixed> $fields_ui
	 * @return array<int,string>
	 */
	private static function ui_allowed(array $fields_ui, string $key, array $fallback): array {
		$f = $fields_ui[$key] ?? null;
		if (! is_array($f)) {
			return $fallback;
		}
		$a = $f['allowed'] ?? null;
		if (! is_array($a) || ! $a) {
			return $fallback;
		}
		return array_values(array_map('strval', $a));
	}

	/**
	 * @param array<string,mixed> $fields_ui
	 */
	private static function ui_maxlen(array $fields_ui, string $key, int $fallback): int {
		$f = $fields_ui[$key] ?? null;
		if (! is_array($f)) {
			return $fallback;
		}
		$ml = $f['maxLen'] ?? null;
		if (! is_numeric($ml)) {
			return $fallback;
		}
		$n = (int) $ml;
		return ($n > 0) ? $n : $fallback;
	}

	/**
	 * scaleFrom ranges can depend on zoomMode.
	 *
	 * @param array<string,mixed> $fields_ui
	 * @return array{min:float,max:float}
	 */
	private static function ui_scale_range(array $fields_ui, string $mode): array {
		$f = $fields_ui['scaleFrom'] ?? null;
		if (! is_array($f)) {
			return [
				'min' => ($mode === 'out') ? 1.01 : 0.85,
				'max' => ($mode === 'out') ? 1.20 : 0.99,
			];
		}

		$min = isset($f['min']) ? (float) $f['min'] : 0.05;
		$max = isset($f['max']) ? (float) $f['max'] : 3.0;

		$modes = $f['modes'] ?? null;
		if (is_array($modes) && is_array($modes[$mode] ?? null)) {
			$mr  = $modes[$mode];
			$min = isset($mr['min']) ? (float) $mr['min'] : $min;
			$max = isset($mr['max']) ? (float) $mr['max'] : $max;
		}

		return ['min' => $min, 'max' => $max];
	}

	// Small sanitizers

	private static function sanitize_key(string $s): string {
		$s = strtolower(trim($s));
		$s = preg_replace('/[^a-z0-9_-]+/', '', $s);
		return is_string($s) ? $s : '';
	}

	/**
	 * Field-aware easing sanitizer (no magic-number coupling).
	 *
	 * @param mixed $val
	 */
	private static function sanitize_easing($val, int $maxlen = 120): string {
		if (! is_string($val) && ! is_numeric($val)) {
			return '';
		}

		$str = trim(wp_strip_all_tags((string) $val));

		if ($maxlen > 0 && mb_strlen($str) > $maxlen) {
			$str = mb_substr($str, 0, $maxlen);
		}

		$str = preg_replace('/[\x00-\x1F\x7F]/u', '', $str);
		if (! is_string($str) || $str === '') {
			return '';
		}

		// Easing may be keywords or function-like values. Keep it strict but practical.
		if (! preg_match('/^[a-zA-Z0-9\-\.\(\),\s]+$/', $str)) {
			WP_InView_Debug::warn('Invalid characters in easing string', ['input' => $str]);
			return '';
		}

		return esc_attr($str);
	}

	/**
	 * Sanitize rootMargin with strict format validation.
	 *
	 * Accepts 1-4 values, each: number with optional decimals and optional unit (px|%).
	 * Examples:
	 *  "0px"
	 *  "0px 0px -10% 0px"
	 *  "-16%"
	 *
	 * @param mixed $val
	 */
	private static function sanitize_rootmargin($val): string {
		if (! is_string($val) && ! is_numeric($val)) {
			return '';
		}

		$str = trim(wp_strip_all_tags((string) $val));

		// Limit length early
		if (mb_strlen($str) > 80) {
			$str = mb_substr($str, 0, 80);
		}

		$str = preg_replace('/[\x00-\x1F\x7F]/u', '', $str);
		if (! is_string($str) || $str === '') {
			return '';
		}

		if (! preg_match('/^-?\d+(\.\d+)?(px|%)?(\s+-?\d+(\.\d+)?(px|%)?){0,3}$/', $str)) {
			WP_InView_Debug::warn('Invalid rootMargin format', ['input' => $str]);
			return '0px';
		}

		return esc_attr($str);
	}

	private static function sanitize_bool($v, bool $fallback): bool {
		if (is_bool($v)) {
			return $v;
		}
		if ($v === 1 || $v === '1' || $v === 'true' || $v === 'yes') {
			return true;
		}
		if ($v === 0 || $v === '0' || $v === 'false' || $v === 'no') {
			return false;
		}
		return $fallback;
	}

	private static function clamp_float($v, float $min, float $max, float $fallback): float {
		if (! is_numeric($v)) {
			return $fallback;
		}
		$n = (float) $v;
		if ($n < $min) {
			return $min;
		}
		if ($n > $max) {
			return $max;
		}
		return $n;
	}

	private static function clamp_int($v, int $min, int $max, int $fallback): int {
		if (! is_numeric($v)) {
			return $fallback;
		}
		$n = (int) $v;
		if ($n < $min) {
			return $min;
		}
		if ($n > $max) {
			return $max;
		}
		return $n;
	}

	// Apply helpers

	/**
	 * @param array<string,string> $data
	 * @param array<string,mixed>  $cfg
	 * @param array<string,mixed>  $fields_ui
	 */
	private static function apply_float(array &$data, array $cfg, array $fields_ui, string $key, string $attr, float $fallback): void {
		if (! array_key_exists($key, $cfg)) {
			return;
		}

		$f   = $fields_ui[$key] ?? null;
		$min = is_array($f) && isset($f['min']) ? (float) $f['min'] : -PHP_FLOAT_MAX;
		$max = is_array($f) && isset($f['max']) ? (float) $f['max'] : PHP_FLOAT_MAX;

		$data[$attr] = (string) self::clamp_float($cfg[$key], $min, $max, $fallback);
	}

	/**
	 * @param array<string,string> $data
	 * @param array<string,mixed>  $cfg
	 * @param array<string,mixed>  $fields_ui
	 */
	private static function apply_int(array &$data, array $cfg, array $fields_ui, string $key, string $attr, int $fallback): void {
		if (! array_key_exists($key, $cfg)) {
			return;
		}

		$f   = $fields_ui[$key] ?? null;
		$min = is_array($f) && isset($f['min']) ? (int) $f['min'] : PHP_INT_MIN;
		$max = is_array($f) && isset($f['max']) ? (int) $f['max'] : PHP_INT_MAX;

		$data[$attr] = (string) self::clamp_int($cfg[$key], $min, $max, $fallback);
	}
}