<?php

declare (strict_types = 1);

if (! defined('ABSPATH')) {
	exit;
}

final class WP_InView_Settings {

	public const OPTION_KEY = 'wp_inview_settings';

	/**
	 * Transient cache key prefix.
	 * Versioned by schema to avoid mismatches after plugin updates.
	 */
	private const CACHE_KEY_PREFIX = 'wp_inview_settings_cache_';

	/**
	 * How long persistent cache should live in production.
	 */
	private const CACHE_DURATION = DAY_IN_SECONDS;

	private const DEBUG_OPT_POSTED   = '_wp_inview_debug_last_posted';
	private const DEBUG_OPT_SANITIZE = '_wp_inview_debug_last_sanitized';

	/**
	 * Per-request cache (safe always).
	 *
	 * @var array<string,mixed>|null
	 */
	private static $cache = null;

	public static function register(): void {
		register_setting('wp_inview', self::OPTION_KEY, [
			'type'              => 'array',
			'sanitize_callback' => [self::class, 'sanitize_callback'],
			'default'           => (array) WP_InView_Schema::defaults(),
		]);

		// Safety net: if option changes via other code paths, still flush caches.
		add_action('update_option_' . self::OPTION_KEY, [self::class, 'flush_cache'], 10, 0);
		add_action('add_option_' . self::OPTION_KEY, [self::class, 'flush_cache'], 10, 0);
		add_action('delete_option_' . self::OPTION_KEY, [self::class, 'flush_cache'], 10, 0);
	}

	/**
	 * @param mixed $raw
	 * @return array<string,mixed>
	 */
	public static function sanitize_callback($raw): array {
		try {
			// Defense in depth: capability gate.
			if (! current_user_can('manage_options')) {
				WP_InView_Debug::error('Unauthorized settings save attempt', [
					'user_id' => get_current_user_id(),
				]);

				$current = get_option(self::OPTION_KEY, null);
				return is_array($current) ? $current : (array) WP_InView_Schema::defaults();
			}

			$posted = is_array($raw) ? $raw : [];

			if (WP_InView_Debug::enabled()) {
				update_option(self::DEBUG_OPT_POSTED, $posted, false);
			}

			$defaults = (array) WP_InView_Schema::defaults();

			// 1) current saved option
			$current = get_option(self::OPTION_KEY, []);
			$current = is_array($current) ? $current : [];

			// 2) normalize current by defaults (fills missing keys, keeps saved values)
			$base = self::deep_merge($defaults, $current);

			// 3) overlay posted values onto base (only changes what was submitted)
			$merged = self::deep_merge($base, $posted);

			if (WP_InView_Debug::enabled()) {
				update_option(self::DEBUG_OPT_SANITIZE, $merged, false);
			}

			$out = (array) WP_InView_Schema::sanitize_settings($merged);

			// Settings changed -> flush caches so frontend/editor see it immediately.
			self::flush_cache();

			return $out;

		} catch (\Throwable $e) {
			WP_InView_Debug::error('Settings sanitize failed', [
				'message' => $e->getMessage(),
				'code'    => $e->getCode(),
				'trace'   => $e->getTraceAsString(),
			]);

			self::flush_cache();
			return (array) WP_InView_Schema::defaults();
		}
	}

	/**
	 * Always returns normalized + sanitized settings.
	 * Dev: no transient cache (fresh computation), but still uses per-request cache unless forced.
	 * Prod: transient cache enabled + per-request cache.
	 *
	 * @return array<string,mixed>
	 */
	public static function get(bool $force_refresh = false): array {
		// Per-request cache.
		if (! $force_refresh && is_array(self::$cache)) {
			return self::$cache;
		}

		// Dev mode: skip transients to avoid stale data while building.
		$use_transient = ! self::is_development();

		$cache_key = self::cache_key();

		if ($use_transient && ! $force_refresh) {
			$cached = get_transient($cache_key);
			if (is_array($cached)) {
				self::$cache = $cached;
				return $cached;
			}
		}

		$raw = get_option(self::OPTION_KEY, null);
		$raw = is_array($raw) ? $raw : [];

		$defaults = (array) WP_InView_Schema::defaults();
		$merged   = self::deep_merge($defaults, $raw);

		$sanitized = (array) WP_InView_Schema::sanitize_settings($merged);

		// Save caches.
		self::$cache = $sanitized;

		if ($use_transient) {
			set_transient($cache_key, $sanitized, self::CACHE_DURATION);
		}

		return $sanitized;
	}

	public static function flush_cache(): void {
		self::$cache = null;

		$cache_key = self::cache_key();

		// Remove transient cache (no-op if not set).
		delete_transient($cache_key);

		// Optional extra safety for persistent object cache setups.
		if (function_exists('wp_cache_delete')) {
			wp_cache_delete($cache_key, 'transient');
			wp_cache_delete($cache_key, 'transient_timeout');
		}
	}

	private static function cache_key(): string {
		$schema_version = class_exists('WP_InView_Schema')
			? (int) WP_InView_Schema::SCHEMA_VERSION
			: 1;

		return self::CACHE_KEY_PREFIX . 'v' . $schema_version;
	}

	private static function is_development(): bool {
		return class_exists('WP_InView_Debug') && WP_InView_Debug::is_development();
	}

	/**
	 * Deep merge overlay into base (overlay wins).
	 *
	 * @param array<string|int,mixed> $base
	 * @param array<string|int,mixed> $overlay
	 * @return array<string|int,mixed>
	 */
	private static function deep_merge(array $base, array $overlay): array {
		foreach ($overlay as $k => $v) {
			if (is_array($v) && isset($base[$k]) && is_array($base[$k])) {
				$base[$k] = self::deep_merge($base[$k], $v);
			} else {
				$base[$k] = $v;
			}
		}

		return $base;
	}
}