<?php

declare (strict_types = 1);

if (! defined('ABSPATH')) {
	exit;
}

trait WP_InView_Schema_UI {

	/**
	 * Returns the full UI schema.
	 *
	 * Static cache intentionally removed — all strings are wrapped in __()
	 * which must be evaluated per-request after load_plugin_textdomain().
	 *
	 * @return array<string,mixed>
	 */
	public static function ui_schema(): array {

		$schema = [
			'animation'      => [
				'info'     => [
					'title'            => __('Animation dashboard', 'wp-inview'),
					'desc'             => __('Global settings and presets for in-view animations.', 'wp-inview'),
					'toggleSwitchHelp' => __('Page transitions are unaffected.', 'wp-inview'),
					'enableLabel'      => __('Enable animations', 'wp-inview'),
					'switchOn'         => __('On', 'wp-inview'),
					'switchOff'        => __('Off', 'wp-inview'),
				],

				'fields'   => [
					'once'          => [
						'label' => __('Animate once', 'wp-inview'),
						'help'  => __('If enabled, the animation runs only the first time the element enters the viewport.', 'wp-inview'),
					],

					'triggerPreset' => [
						'label'   => __('Trigger', 'wp-inview'),
						'help'    => __('Pick a trigger profile. Custom lets you set threshold and root margin manually.', 'wp-inview'),
						'allowed' => ['natural', 'early_bird', 'on_focus', 'just_in_time', 'late_entrance', 'custom'],
					],

					'direction'     => [
						'label'   => __('Direction', 'wp-inview'),
						'help'    => __('Where the element comes from.', 'wp-inview'),
						'allowed' => ['up', 'down', 'left', 'right'],
						'chips'   => [
							['label' => __('Up', 'wp-inview'), 'value' => 'up'],
							['label' => __('Down', 'wp-inview'), 'value' => 'down'],
							['label' => __('Left', 'wp-inview'), 'value' => 'left'],
							['label' => __('Right', 'wp-inview'), 'value' => 'right'],
						],
					],

					'distance'      => [
						'label' => __('Distance', 'wp-inview'),
						'help'  => __('How far the element moves (px).', 'wp-inview'),
						'min'   => 0,
						'max'   => 200,
						'step'  => 1,
						'unit'  => 'px',
					],

					'opacityFrom'   => [
						'label'       => __('Opacity from', 'wp-inview'),
						'help'        => __('Pick a start transparency. Custom unlocks manual value.', 'wp-inview'),
						'min'         => 0,
						'max'         => 1,
						'step'        => 0.05,
						'chips'       => [
							['label' => __('Invisible start', 'wp-inview'), 'value' => '0'],
							['label' => __('Soft fade', 'wp-inview'), 'value' => '0.2'],
							['label' => __('Half ghost', 'wp-inview'), 'value' => '0.5'],
							['label' => __('Almost there', 'wp-inview'), 'value' => '0.8'],
							['label' => __('Fully visible', 'wp-inview'), 'value' => '1.0'],
							['label' => __('Custom…', 'wp-inview'), 'value' => 'custom'],
						],
						'placeholder' => '0.35',
						'example'     => '0.35',
					],

					'duration'      => [
						'label' => __('Duration', 'wp-inview'),
						'help'  => __('Animation time (ms).', 'wp-inview'),
						'min'   => 100,
						'max'   => 6000,
						'step'  => 50,
						'unit'  => 'ms',
					],

					'delay'         => [
						'label' => __('Delay', 'wp-inview'),
						'help'  => __('Wait before starting (ms).', 'wp-inview'),
						'min'   => 0,
						'max'   => 5000,
						'step'  => 50,
						'unit'  => 'ms',
					],

					'easing'        => [
						'label'       => __('Easing', 'wp-inview'),
						'help'        => __('Pick a feel. Custom unlocks the curve input.', 'wp-inview'),
						'placeholder' => 'cubic-bezier(0.22, 1, 0.36, 1)',
						'maxLen'      => 120,
						'customLabel' => __('Custom easing', 'wp-inview'),
						'customHelp'  => __('CSS cubic-bezier value, e.g. cubic-bezier(0.22, 1, 0.36, 1).', 'wp-inview'),
					],

					'threshold'     => [
						'label' => __('Trigger threshold', 'wp-inview'),
						'help'  => __('How much needs to be visible (0–1).', 'wp-inview'),
						'min'   => 0,
						'max'   => 1,
						'step'  => 0.05,
					],

					'rootMargin'    => [
						'label'       => __('Root margin', 'wp-inview'),
						'help'        => __('Adjusts how early the animation can trigger. Negative bottom starts earlier.', 'wp-inview'),
						'example'     => '0px 0px -4% 0px',
						'placeholder' => '0px 0px -4% 0px',
						'maxLen'      => 80,
					],

					'zoomMode'      => [
						'label'   => __('Zoom mode', 'wp-inview'),
						'help'    => __('Zoom in starts below 1, Zoom out starts above 1. Always ends at 1.', 'wp-inview'),
						'allowed' => ['in', 'out'],
						'chips'   => [
							['label' => __('Zoom in', 'wp-inview'), 'value' => 'in'],
							['label' => __('Zoom out', 'wp-inview'), 'value' => 'out'],
						],
					],

					'scaleFrom'     => [
						'label' => __('Scale start', 'wp-inview'),
						'help'  => __('Shown as percent. Stored as scale. Example: 92% equals 0.92.', 'wp-inview'),
						'min'   => 0.05,
						'max'   => 3.0,
						'step'  => 0.01,
						'modes' => [
							'in'  => ['min' => 0.75, 'max' => 0.99],
							'out' => ['min' => 1.01, 'max' => 1.25],
						],
						'unit'  => '%',
					],

					'rubberMode'    => [
						'label'   => __('Rubber', 'wp-inview'),
						'help'    => __('Adds a small overshoot at the end. Custom unlocks amplitude.', 'wp-inview'),
						'allowed' => ['off', 'soft', 'medium', 'hard', 'custom'],
						'chips'   => [
							['label' => __('Off', 'wp-inview'), 'value' => 'off'],
							['label' => __('Soft', 'wp-inview'), 'value' => 'soft'],
							['label' => __('Medium', 'wp-inview'), 'value' => 'medium'],
							['label' => __('Hard', 'wp-inview'), 'value' => 'hard'],
							['label' => __('Custom…', 'wp-inview'), 'value' => 'custom'],
						],
					],

					'rubberAmp'     => [
						'label'       => __('Rubber amplitude', 'wp-inview'),
						'help'        => __('How strong the overshoot is. Example: 3.5% equals 0.035.', 'wp-inview'),
						'min'         => 0,
						'max'         => 0.1,
						'step'        => 0.005,
						'chips'       => [
							['label' => __('Soft', 'wp-inview'), 'value' => '0.02'],
							['label' => __('Medium', 'wp-inview'), 'value' => '0.035'],
							['label' => __('Hard', 'wp-inview'), 'value' => '0.055'],
						],
						'example'     => '0.035',
						'placeholder' => '0.035',
					],

					'blurFrom'      => [
						'label' => __('Blur from', 'wp-inview'),
						'help'  => __('Start blur strength (px). Ends at 0.', 'wp-inview'),
						'min'   => 0,
						'max'   => 250,
						'step'  => 1,
						'unit'  => 'px',
					],

					'maskVariant'   => [
						'label'   => __('Mask variant', 'wp-inview'),
						'help'    => __('Wipe reveals content in one direction. Radial reveals from center.', 'wp-inview'),
						'allowed' => ['wipe'],
						'chips'   => [
							['label' => __('Wipe', 'wp-inview'), 'value' => 'wipe'],
						],
						'maxLen'  => 32,
					],

					'maskDirection' => [
						'label'   => __('Direction', 'wp-inview'),
						'help'    => __('Reveal direction for wipe.', 'wp-inview'),
						'allowed' => ['up', 'down', 'left', 'right', 'horizontal', 'vertical'],
						'chips'   => [
							['label' => __('Up', 'wp-inview'), 'value' => 'up'],
							['label' => __('Down', 'wp-inview'), 'value' => 'down'],
							['label' => __('Left', 'wp-inview'), 'value' => 'left'],
							['label' => __('Right', 'wp-inview'), 'value' => 'right'],
							['label' => __('Horizontal', 'wp-inview'), 'value' => 'horizontal'],
							['label' => __('Vertical', 'wp-inview'), 'value' => 'vertical'],
						],
						'maxLen'  => 32,
					],

					'maskFeather'   => [
						'label' => __('Feather', 'wp-inview'),
						'help'  => __('Softness of the mask edge in pixels.', 'wp-inview'),
						'min'   => 0,
						'max'   => 250,
						'step'  => 1,
					],

					'maskInset'     => [
						'label' => __('Inset', 'wp-inview'),
						'help'  => __('How much is hidden at start, in percent.', 'wp-inview'),
						'min'   => 5,
						'max'   => 100,
						'step'  => 1,
					],
				],

				'copy'     => [
					'scaleFromHint'     => __('Shown as percent. Stored as scale. Example: 92% equals 0.92.', 'wp-inview'),
					'easingTip'         => __('Tip: if you don\'t know what to pick, choose Smooth.', 'wp-inview'),
					'rubberAmpHint'     => __('Example: 3.5% equals 0.035.', 'wp-inview'),
					'customEasing'      => __('Custom easing', 'wp-inview'),
					'customColor'       => __('Custom color', 'wp-inview'),
					'customOpacity'     => __('Custom opacity', 'wp-inview'),
					'customLabel'       => __('Custom', 'wp-inview'),
					'customEllipsis'    => __('Custom…', 'wp-inview'),
					'resetSettings'     => __('Reset settings to defaults', 'wp-inview'),
					'switchOn'          => __('On', 'wp-inview'),
					'switchOff'         => __('Off', 'wp-inview'),
					// Used in preset-zoom.php: "Saved as 0.92" readout below scale input.
					'savedAs'           => __('Saved as', 'wp-inview'),
					// Fallback card description when registry['description'] is empty.
					// {preset} is replaced with the preset key at render time.
					'presetCardDescTpl' => __('Applies to blocks using preset {preset}.', 'wp-inview'),
				],

				'speeds'   => [
					['key' => 'fast', 'label' => __('Fast', 'wp-inview'), 'apply' => ['duration' => 450, 'delay' => 0]],
					['key' => 'regular', 'label' => __('Regular', 'wp-inview'), 'apply' => ['duration' => 650, 'delay' => 0]],
					['key' => 'slow', 'label' => __('Slow', 'wp-inview'), 'apply' => ['duration' => 900, 'delay' => 50]],
					['key' => 'custom', 'label' => __('Custom', 'wp-inview'), 'type' => 'custom'],
				],

				'easing'   => [
					['key' => 'smooth', 'label' => __('Smooth', 'wp-inview'), 'value' => 'cubic-bezier(0.12, 1, 0.42, 1)', 'type' => 'preset'],
					['key' => 'easeOut', 'label' => __('Ease out', 'wp-inview'), 'value' => 'cubic-bezier(0.22, 1, 0.36, 1)', 'type' => 'preset'],
					['key' => 'snappy', 'label' => __('Snappy', 'wp-inview'), 'value' => 'cubic-bezier(0.34, 1.56, 0.64, 1)', 'type' => 'preset'],
					['key' => 'easeIO', 'label' => __('Ease in-out', 'wp-inview'), 'value' => 'cubic-bezier(0.37, 0, 0.63, 1)', 'type' => 'preset'],
					['key' => 'linear', 'label' => __('Linear', 'wp-inview'), 'value' => 'cubic-bezier(0, 0, 1, 1)', 'type' => 'preset'],
					['key' => 'custom', 'label' => __('Custom…', 'wp-inview'), 'value' => 'cubic-bezier(0.22, 1, 0.36, 1)', 'type' => 'custom'],
				],

				'triggers' => [[
					'key'        => 'early_bird',
					'label'      => __('Early bird', 'wp-inview'),
					'desc'       => __('Starts a bit earlier, feels snappier.', 'wp-inview'),
					'threshold'  => 0,
					'rootMargin' => '0px 0px 4% 0px',
					'type'       => 'preset',
				], [
					'key'        => 'just_in_time',
					'label'      => __('Just in time', 'wp-inview'),
					'desc'       => __('Triggers exactly on entry, very predictable.', 'wp-inview'),
					'threshold'  => 0,
					'rootMargin' => '0px',
					'type'       => 'preset',
				],
					[
						'key'        => 'natural',
						'label'      => __('Natural', 'wp-inview'),
						'desc'       => __('Clear entry into viewport, best default.', 'wp-inview'),
						'threshold'  => 0.0,
						'rootMargin' => '0px 0px -2% 0px',
						'type'       => 'preset',
					],

					[
						'key'        => 'on_focus',
						'label'      => __('On focus', 'wp-inview'),
						'desc'       => __('Triggers when it is more in view.', 'wp-inview'),
						'threshold'  => 0.15,
						'rootMargin' => '0px 0px -4% 0px',
						'type'       => 'preset',
					],

					[
						'key'        => 'late_entrance',
						'label'      => __('Late entrance', 'wp-inview'),
						'desc'       => __('Waits longer, more subtle.', 'wp-inview'),
						'threshold'  => 0.15,
						'rootMargin' => '0px 0px -10% 0px',
						'type'       => 'preset',
					],
					[
						'key'        => 'custom',
						'label'      => __('Custom…', 'wp-inview'),
						'desc'       => __('Set values manually.', 'wp-inview'),
						'threshold'  => 0.15,
						'rootMargin' => '0px 0px -4% 0px',
						'type'       => 'custom',
					],
				],

				'cascade'  => [
					'info'   => [
						'title' => __('Cascade', 'wp-inview'),
						'desc'  => __('Cascade staggers child animations. Enable Cascade per-block in the sidebar, global settings define which children are affected.', 'wp-inview'),
					],
					'copy'   => [
						'resetSettings' => __('Reset settings to defaults', 'wp-inview'),
					],
					'fields' => [
						'selector' => [
							'label'       => __('Child selector', 'wp-inview'),
							'help'        => __('Target elements by tag or class. Leave empty to auto-detect.', 'wp-inview'),
							'placeholder' => __('img, li, .your-class', 'wp-inview'),
							'docs'        => 'https://docs.wp-inview.com/cascade#selector',
							'maxLen'      => 120,
						],
						'limit'    => [
							'label' => __('Limit', 'wp-inview'),
							'help'  => __('Limits how many children animate. Protects performance on large containers.', 'wp-inview'),
							'min'   => 1,
							'max'   => 200,
							'step'  => 1,
						],
						'step'     => [
							'label' => __('Step', 'wp-inview'),
							'help'  => __('Delay between each child (ms). Overridden by the block\'s own delay value when set.', 'wp-inview'),
							'min'   => 0,
							'max'   => 2000,
							'step'  => 10,
						],
					],
				],
			],

			'pageTransition' => [
				'info'    => [
					'title'            => __('Page Transition', 'wp-inview'),
					'desc'             => __('Your visitors will experience seamless navigation similar to modern single-page applications.', 'wp-inview'),
					'enableLabel'      => __('Enable page transitions', 'wp-inview'),
					'toggleSwitchHelp' => __('Smooth animated transitions between pages.', 'wp-inview'),
					'activeLabel'      => __('Transition effect', 'wp-inview'),
					'activeLabelHelp'  => __('Choose the animation style for page transitions.', 'wp-inview'),
					'switchOn'         => __('On', 'wp-inview'),
					'switchOff'        => __('Off', 'wp-inview'),
				],

				'copy'    => [
					'variantSettingsSuffix' => __('Settings', 'wp-inview'),
					'customEasing'          => __('Custom easing', 'wp-inview'),
					'customEasingHelp'      => __('CSS cubic-bezier, e.g. cubic-bezier(0.22, 1, 0.36, 1)', 'wp-inview'),
					'customColor'           => __('Custom color', 'wp-inview'),
					'customOpacity'         => __('Custom opacity', 'wp-inview'),
					'customOpacityHelp'     => __('Value between 0 (transparent) and 1 (solid).', 'wp-inview'),
					'customLabel'           => __('Custom', 'wp-inview'),
					'curtainDirectionHelp'  => __('The curtain slides in from this side.', 'wp-inview'),
					'curtainColorHelp'      => __('Color of the curtain overlay.', 'wp-inview'),
				],

				'fields'  => [
					'activeVariant'  => [
						'label'   => __('Active transition', 'wp-inview'),
						'help'    => __('Choose which transition effect to use.', 'wp-inview'),
						'allowed' => ['fade', 'blur', 'curtain'],
						'chips'   => [
							['label' => __('Fade', 'wp-inview'), 'value' => 'fade'],
							['label' => __('Blur', 'wp-inview'), 'value' => 'blur'],
							['label' => __('Curtain', 'wp-inview'), 'value' => 'curtain'],
						],
					],

					'duration'       => [
						'label' => __('Duration', 'wp-inview'),
						'help'  => __('Transition time in milliseconds.', 'wp-inview'),
						'min'   => 100,
						'max'   => 2000,
						'step'  => 50,
						'unit'  => 'ms',
					],

					'easing'         => [
						'label'       => __('Easing', 'wp-inview'),
						'help'        => __('Choose the timing function or enter custom cubic-bezier.', 'wp-inview'),
						'placeholder' => 'cubic-bezier(0.22, 1, 0.36, 1)',
						'maxLen'      => 120,
					],

					'overlayColor'   => [
						'label'       => __('Overlay color', 'wp-inview'),
						'help'        => __('Color of the transition overlay.', 'wp-inview'),
						'placeholder' => '#ffffff',
						'maxLen'      => 9,
					],

					'overlayOpacity' => [
						'label' => __('Overlay opacity', 'wp-inview'),
						'help'  => __('Opacity of the overlay (0–1).', 'wp-inview'),
						'min'   => 0,
						'max'   => 1,
						'step'  => 0.05,
					],

					'blurAmount'     => [
						'label' => __('Blur amount', 'wp-inview'),
						'help'  => __('Blur intensity in pixels (for blur variant).', 'wp-inview'),
						'min'   => 0,
						'max'   => 500,
						'step'  => 1,
						'unit'  => 'px',
					],

					'direction'      => [
						'label'   => __('Curtain direction', 'wp-inview'),
						'help'    => __('Direction the curtain slides from.', 'wp-inview'),
						'allowed' => ['left', 'right', 'top', 'bottom', 'center'],
						'chips'   => [
							['label' => __('Left', 'wp-inview'), 'value' => 'left'],
							['label' => __('Right', 'wp-inview'), 'value' => 'right'],
							['label' => __('Top', 'wp-inview'), 'value' => 'top'],
							['label' => __('Bottom', 'wp-inview'), 'value' => 'bottom'],
							['label' => __('Center', 'wp-inview'), 'value' => 'center'],
						],
					],

					'splitMode'      => [
						'label' => __('Split mode', 'wp-inview'),
						'help'  => __('Divide curtain into two panels from center.', 'wp-inview'),
					],
				],

				'easing'  => [
					['key' => 'smooth', 'label' => __('Smooth', 'wp-inview'), 'value' => 'cubic-bezier(0.12, 1, 0.42, 1)', 'type' => 'preset'],
					['key' => 'easeOut', 'label' => __('Ease out', 'wp-inview'), 'value' => 'cubic-bezier(0.22, 1, 0.36, 1)', 'type' => 'preset'],
					['key' => 'snappy', 'label' => __('Snappy', 'wp-inview'), 'value' => 'cubic-bezier(0.34, 1.56, 0.64, 1)', 'type' => 'preset'],
					['key' => 'easeIO', 'label' => __('Ease in-out', 'wp-inview'), 'value' => 'cubic-bezier(0.37, 0, 0.63, 1)', 'type' => 'preset'],
					['key' => 'linear', 'label' => __('Linear', 'wp-inview'), 'value' => 'cubic-bezier(0, 0, 1, 1)', 'type' => 'preset'],
					['key' => 'custom', 'label' => __('Custom…', 'wp-inview'), 'value' => 'cubic-bezier(0.22, 1, 0.36, 1)', 'type' => 'custom'],
				],

				'colors'  => [
					['label' => __('White', 'wp-inview'), 'value' => '#ffffff'],
					['label' => __('Black', 'wp-inview'), 'value' => '#000000'],
					['label' => __('Custom…', 'wp-inview'), 'value' => 'custom'],
				],

				'opacity' => [
					['label' => __('Transparent', 'wp-inview'), 'value' => '0'],
					['label' => __('Light', 'wp-inview'), 'value' => '0.5'],
					['label' => __('Medium', 'wp-inview'), 'value' => '0.8'],
					['label' => __('Solid', 'wp-inview'), 'value' => '1'],
					['label' => __('Custom…', 'wp-inview'), 'value' => 'custom'],
				],
			],

			'tabs'           => [
				'animation'  => __('Animation', 'wp-inview'),
				'cascade'    => __('Cascade', 'wp-inview'),
				'transition' => __('Page transition', 'wp-inview'),
				'settings'   => __('Settings', 'wp-inview'),
			],
		];

		// Inject triggerPreset chips dynamically from triggers list
		$triggers = $schema['animation']['triggers'] ?? [];
		$chips    = [];
		foreach ($triggers as $row) {
			if (! is_array($row)) {
				continue;
			}
			$key   = isset($row['key']) ? (string) $row['key'] : '';
			$label = isset($row['label']) ? (string) $row['label'] : '';
			if ($key === '' || $label === '') {
				continue;
			}
			$chips[] = ['label' => $label, 'value' => $key];
		}
		if ($chips) {
			$schema['animation']['fields']['triggerPreset']['chips'] = $chips;
		}

		return $schema;
	}
}