<?php

declare (strict_types = 1);

if (! defined('ABSPATH')) {
	exit;
}

trait WP_InView_Schema_Normalize {

	/**
	 * If triggerPreset points to a preset trigger (not custom), override threshold/rootMargin.
	 *
	 * @return array{0: float|null, 1: string|null}
	 */
	private static function sanitize_cascade_selector(string $s): string {
		$s = trim($s);
		if ($s === '') {
			return '';
		}

		$ok = preg_match('/^[a-zA-Z0-9\\s,>+~.#:*\\[\\]\\(\\)\\^$|=\'\\"_\\-]+$/', $s) === 1;
		if (! $ok) {
			return '';
		}

		return $s;
	}

	private static function resolve_trigger_override(array $src): array {
		$trigger = isset($src['triggerPreset']) ? trim((string) $src['triggerPreset']) : '';
		if ($trigger === '' || $trigger === 'custom') {
			return [null, null];
		}

		$ui   = self::ui_schema();
		$anim = is_array($ui['animation'] ?? null) ? $ui['animation'] : [];
		$rows = $anim['triggers'] ?? [];

		if (! is_array($rows)) {
			return [null, null];
		}

		foreach ($rows as $row) {
			if (! is_array($row)) {
				continue;
			}

			$type = (string) ($row['type'] ?? '');
			$key  = (string) ($row['key'] ?? '');

			if ($type !== 'preset' || $key !== $trigger) {
				continue;
			}

			$thr = array_key_exists('threshold', $row) ? (float) $row['threshold'] : null;
			$rm  = array_key_exists('rootMargin', $row) ? (string) $row['rootMargin'] : null;

			return [$thr, $rm];
		}

		return [null, null];
	}

	private static function normalize_preset_settings(string $presetKey, array $src, array $fallback = []): array {
		$registry = self::presets_registry();
		$def      = $registry[$presetKey] ?? null;
		$fields   = is_array($def['fields'] ?? null) ? $def['fields'] : [];

		$out = [];

		[$thresholdOverride, $rootMarginOverride] = self::resolve_trigger_override($src);

		foreach ($fields as $fieldKey => $fieldDef) {
			$type    = (string) ($fieldDef['type'] ?? 'string');
			$default = $fieldDef['default'] ?? null;

			$val = array_key_exists($fieldKey, $src)
				? $src[$fieldKey]
				: (array_key_exists($fieldKey, $fallback) ? $fallback[$fieldKey] : $default);

			// Trigger preset overrides (UI helper field must be preserved in registry)
			if ($fieldKey === 'threshold' && $thresholdOverride !== null) {
				$val = $thresholdOverride;
			}
			if ($fieldKey === 'rootMargin' && $rootMarginOverride !== null) {
				$val = $rootMarginOverride;
			}

			switch ($type) {
			case 'bool':
				$out[$fieldKey] = (bool) $val;
				break;

			case 'int':{
					$ui             = self::ui_field($fieldKey);
					$min            = isset($ui['min']) ? (int) $ui['min'] : PHP_INT_MIN;
					$max            = isset($ui['max']) ? (int) $ui['max'] : PHP_INT_MAX;
					$out[$fieldKey] = self::clamp_int($val, $min, $max);
					break;
				}

			case 'float':{
					$ui             = self::ui_field($fieldKey);
					$min            = isset($ui['min']) ? (float) $ui['min'] : -PHP_FLOAT_MAX;
					$max            = isset($ui['max']) ? (float) $ui['max'] : PHP_FLOAT_MAX;
					$out[$fieldKey] = self::clamp_float($val, $min, $max);
					break;
				}

			default: {
					$defaultStr = is_string($default) ? $default : (string) ($default ?? '');
					$s          = trim(is_string($val) ? $val : ($val === null ? '' : (string) $val));

					// SECURITY FIX #2: Strict validation based on field type
					switch ($fieldKey) {
					case 'easing':
						// Validate easing format: alphanumeric, dash, dot, parens, comma
						if ($s !== '' && ! preg_match('/^[a-zA-Z0-9\\-\\.\\(\\),\\s]+$/', $s)) {
							WP_InView_Debug::warn('Invalid easing value rejected', [
								'field' => $fieldKey,
								'value' => $s,
							]);
							$s = $defaultStr;
						}
						break;

					case 'rootMargin':
						// Validate rootMargin format: CSS margin syntax
						if ($s !== '' && ! preg_match('/^-?\\d+(\\.\\d+)?(px|%)(\\s+-?\\d+(\\.\\d+)?(px|%)?){0,3}$/', $s)) {
							WP_InView_Debug::warn('Invalid rootMargin value rejected', [
								'field' => $fieldKey,
								'value' => $s,
							]);
							$s = '0px 0px -10% 0px'; // Safe default
						}
						break;

					case 'direction':
						// Validate direction: only allowed values
						$allowed = ['up', 'down', 'left', 'right'];
						if ($s !== '' && ! in_array($s, $allowed, true)) {
							WP_InView_Debug::warn('Invalid direction value rejected', [
								'field'   => $fieldKey,
								'value'   => $s,
								'allowed' => $allowed,
							]);
							$s = $defaultStr;
						}
						break;

					case 'maskDirection':
						// Validate mask direction
						$allowed = ['up', 'down', 'left', 'right', 'horizontal', 'vertical'];
						if ($s !== '' && ! in_array($s, $allowed, true)) {
							$s = $defaultStr;
						}
						break;

					case 'maskVariant':
						// Validate mask variant
						$allowed = ['wipe'];
						if ($s !== '' && ! in_array($s, $allowed, true)) {
							$s = $defaultStr;
						}
						break;

					case 'zoomMode':
						// Validate zoom mode
						$allowed = ['in', 'out'];
						if ($s !== '' && ! in_array($s, $allowed, true)) {
							$s = $defaultStr;
						}
						break;

					case 'rubberMode':
						// Validate rubber mode
						$allowed = ['off', 'soft', 'medium', 'hard', 'custom'];
						if ($s !== '' && ! in_array($s, $allowed, true)) {
							WP_InView_Debug::warn('Invalid rubberMode value rejected', [
								'field'   => $fieldKey,
								'value'   => $s,
								'allowed' => $allowed,
							]);
							$s = $defaultStr;
						}
						break;

					case 'cascadeSelector':
						$s = self::sanitize_cascade_selector($s);
						break;
					}

					// Apply max length from UI schema
					$maxlen = isset($fieldDef['maxLen']) ? (int) $fieldDef['maxLen'] : 200;
					if (mb_strlen($s) > $maxlen) {
						$s = mb_substr($s, 0, $maxlen);
					}

					$out[$fieldKey] = $s !== '' ? $s : $defaultStr;
					break;
				}
			}
		}

		return $out;
	}

	/**
	 * Sanitize a single page transition variant's fields against its registry definition.
	 *
	 * @param string $variantKey
	 * @param array  $src        Raw input for this variant.
	 * @param array  $defaults   Default values for this variant.
	 * @return array
	 */
	private static function normalize_pt_variant(string $variantKey, array $src, array $defaults): array {
		$registry   = self::page_transition_registry();
		$variantDef = $registry[$variantKey] ?? null;
		$fields     = is_array($variantDef['fields'] ?? null) ? $variantDef['fields'] : [];

		$out = [];

		// Preserve 'enabled' per-variant (bool, not in fields registry)
		$out['enabled'] = isset($src['enabled']) ? (bool) $src['enabled'] : (bool) ($defaults['enabled'] ?? false);

		foreach ($fields as $fieldKey => $fieldDef) {
			$type    = (string) ($fieldDef['type'] ?? 'string');
			$default = $fieldDef['default'] ?? null;

			$val = array_key_exists($fieldKey, $src)
				? $src[$fieldKey]
				: (array_key_exists($fieldKey, $defaults) ? $defaults[$fieldKey] : $default);

			switch ($type) {
			case 'bool':
				$out[$fieldKey] = (bool) $val;
				break;

			case 'int':{
					$ui             = self::ui_pt_field($fieldKey);
					$min            = isset($ui['min']) ? (int) $ui['min'] : PHP_INT_MIN;
					$max            = isset($ui['max']) ? (int) $ui['max'] : PHP_INT_MAX;
					$out[$fieldKey] = self::clamp_int($val, $min, $max);
					break;
				}

			case 'float':{
					$ui             = self::ui_pt_field($fieldKey);
					$min            = isset($ui['min']) ? (float) $ui['min'] : -PHP_FLOAT_MAX;
					$max            = isset($ui['max']) ? (float) $ui['max'] : PHP_FLOAT_MAX;
					$out[$fieldKey] = self::clamp_float($val, $min, $max);
					break;
				}

			default: {
					$defaultStr = is_string($default) ? $default : (string) ($default ?? '');
					$s          = trim(is_string($val) ? $val : ($val === null ? '' : (string) $val));

					switch ($fieldKey) {
					case 'easing':
						if ($s !== '' && ! preg_match('/^[a-zA-Z0-9\\-\\.\\(\\),\\s]+$/', $s)) {
							$s = $defaultStr;
						}
						break;

					case 'overlayColor':
						// Allow hex colors (#rgb, #rgba, #rrggbb, #rrggbbaa) and CSS variables
						if ($s !== '' && ! preg_match('/^(#([0-9a-fA-F]{3}|[0-9a-fA-F]{4}|[0-9a-fA-F]{6}|[0-9a-fA-F]{8})|var\\(--[a-zA-Z0-9-]+\\))$/', $s)) {
							$s = $defaultStr;
						}
						break;

					case 'direction':
						$allowed = ['left', 'right', 'top', 'bottom', 'center'];
						if ($s !== '' && ! in_array($s, $allowed, true)) {
							$s = $defaultStr;
						}
						break;
					}

					$ui_field = self::ui_pt_field($fieldKey);
					$maxlen   = isset($ui_field['maxLen']) ? (int) $ui_field['maxLen'] : 200;
					if (mb_strlen($s) > $maxlen) {
						$s = mb_substr($s, 0, $maxlen);
					}

					$out[$fieldKey] = $s !== '' ? $s : $defaultStr;
					break;
				}
			}
		}

		return $out;
	}

	public static function normalize_settings(array $src): array {

		$defaults = self::defaults();

		// Base output = defaults (single source of truth)
		$out = is_array($defaults) ? $defaults : [];

		// Force current schema version
		$out['schemaVersion'] = (int) self::SCHEMA_VERSION;

		// Hard safety in case defaults were modified/partial
		if (! isset($out['animation']) || ! is_array($out['animation'])) {
			$out['animation'] = [
				'settings' => [],
				'presets'  => [],
				'cascade'  => [],
			];
		}
		if (! isset($out['animation']['settings']) || ! is_array($out['animation']['settings'])) {
			$out['animation']['settings'] = [];
		}
		if (! isset($out['animation']['presets']) || ! is_array($out['animation']['presets'])) {
			$out['animation']['presets'] = [];
		}
		if (! isset($out['animation']['cascade']) || ! is_array($out['animation']['cascade'])) {
			$out['animation']['cascade'] = [];
		}
		if (! isset($out['pageTransition']) || ! is_array($out['pageTransition'])) {
			$out['pageTransition'] = [];
		}

		// UI (admin-only settings)
		$uiSrc = $src['ui'] ?? [];
		$uiSrc = is_array($uiSrc) ? $uiSrc : [];

		$allowedLangs = ['auto', 'en', 'pl', 'es'];
		$langDefault  = (string) ($defaults['ui']['language'] ?? 'auto');
		$langRaw      = isset($uiSrc['language']) ? trim((string) $uiSrc['language']) : $langDefault;

		$out['ui']['language'] = in_array($langRaw, $allowedLangs, true) ? $langRaw : $langDefault;

		$allowedModes = ['popup', 'sidebar'];
		$modeDefault  = (string) ($defaults['ui']['editorPanelMode'] ?? 'popup');
		$modeRaw      = isset($uiSrc['editorPanelMode']) ? trim((string) $uiSrc['editorPanelMode']) : $modeDefault;

		$out['ui']['editorPanelMode'] = in_array($modeRaw, $allowedModes, true) ? $modeRaw : $modeDefault;

		$badgesDefault                   = (bool) ($defaults['ui']['showListViewBadges'] ?? true);
		$badgesRaw                       = isset($uiSrc['showListViewBadges']) ? (bool) $uiSrc['showListViewBadges'] : $badgesDefault;
		$out['ui']['showListViewBadges'] = $badgesRaw;

		// Animation Settings (global)
		$animSrc = $src['animation'] ?? [];
		$animSrc = is_array($animSrc) ? $animSrc : [];

		$animSettingsSrc = $animSrc['settings'] ?? [];
		$animSettingsSrc = is_array($animSettingsSrc) ? $animSettingsSrc : [];

		$animDefaults       = is_array($defaults['animation']['settings'] ?? null) ? $defaults['animation']['settings'] : [];
		$animEnabledDefault = (bool) ($animDefaults['enabled'] ?? true);

		// Checkbox pattern: '1' or '0'. If missing, keep default.
		$animEnabledRaw = $animSettingsSrc['enabled'] ?? null;

		if ($animEnabledRaw === null) {
			$animEnabled = $animEnabledDefault;
		} else {
			$animEnabled = ((string) $animEnabledRaw === '1');
		}

		$out['animation']['settings']['enabled'] = $animEnabled;

		// Presets
		$presetsSrc = $src['animation']['presets'] ?? [];
		if (! is_array($presetsSrc)) {
			$presetsSrc = [];
		}

		foreach (self::presets_registry() as $presetKey => $_presetDef) {
			$presetKey = (string) $presetKey;

			$srcPreset = is_array($presetsSrc[$presetKey] ?? null) ? $presetsSrc[$presetKey] : [];
			$fallback  = is_array($defaults['animation']['presets'][$presetKey] ?? null) ? $defaults['animation']['presets'][$presetKey] : [];

			$out['animation']['presets'][$presetKey] = self::normalize_preset_settings(
				$presetKey,
				$srcPreset,
				$fallback
			);
		}

		// Cascade (global)
		$cascadeSrc = $src['animation']['cascade'] ?? [];
		if (! is_array($cascadeSrc)) {
			$cascadeSrc = [];
		}

		$ui        = self::ui_schema();
		$animUi    = is_array($ui['animation'] ?? null) ? $ui['animation'] : [];
		$casBlock  = is_array($animUi['cascade'] ?? null) ? $animUi['cascade'] : [];
		$cascadeUi = is_array($casBlock['fields'] ?? null) ? $casBlock['fields'] : [];

		$defaultsCascade = is_array($defaults['animation']['cascade'] ?? null) ? $defaults['animation']['cascade'] : [];

		$limitMin = isset($cascadeUi['limit']['min']) ? (int) $cascadeUi['limit']['min'] : 1;
		$limitMax = isset($cascadeUi['limit']['max']) ? (int) $cascadeUi['limit']['max'] : 200;

		$stepMin = isset($cascadeUi['step']['min']) ? (int) $cascadeUi['step']['min'] : 0;
		$stepMax = isset($cascadeUi['step']['max']) ? (int) $cascadeUi['step']['max'] : 2000;

		$limitDefault = (int) ($defaultsCascade['limit'] ?? 12);
		$stepDefault  = (int) ($defaultsCascade['step'] ?? 120);

		$limit = isset($cascadeSrc['limit'])
			? self::clamp_int($cascadeSrc['limit'], $limitMin, $limitMax)
			: $limitDefault;

		$step = isset($cascadeSrc['step'])
			? self::clamp_int($cascadeSrc['step'], $stepMin, $stepMax)
			: $stepDefault;

		$selectorRaw = isset($cascadeSrc['selector']) ? (string) $cascadeSrc['selector'] : '';
		$selectorRaw = trim($selectorRaw);

		$selectorMaxLen = isset($cascadeUi['selector']['maxLen']) ? (int) $cascadeUi['selector']['maxLen'] : 120;
		$selectorRaw    = self::sanitize_text_maxlen($selectorRaw, $selectorMaxLen);
		$selector       = self::sanitize_cascade_selector($selectorRaw);

		$out['animation']['cascade'] = [
			'step'     => $step,
			'limit'    => $limit,
			'selector' => $selector,
		];

		// Page Transition
		$pt = $src['pageTransition'] ?? [];
		if (! is_array($pt)) {
			$pt = [];
		}

		$ptDefaults = is_array($defaults['pageTransition'] ?? null) ? $defaults['pageTransition'] : [];

		// enabled (global toggle)
		$enabledDefault = (bool) ($ptDefaults['enabled'] ?? false);
		$enabled        = isset($pt['enabled']) ? (bool) $pt['enabled'] : $enabledDefault;

		// activeVariant
		$activeVariantDefault = (string) ($ptDefaults['activeVariant'] ?? 'fade');
		$activeVariantAllowed = array_keys(self::page_transition_registry());
		$activeVariantRaw     = isset($pt['activeVariant']) ? trim((string) $pt['activeVariant']) : $activeVariantDefault;
		$activeVariant        = self::sanitize_enum_value($activeVariantAllowed, $activeVariantRaw, $activeVariantDefault);

		// variants — iterate all registry variants
		$ptVariantsSrc      = isset($pt['variants']) && is_array($pt['variants']) ? $pt['variants'] : [];
		$ptVariantsDefaults = isset($ptDefaults['variants']) && is_array($ptDefaults['variants']) ? $ptDefaults['variants'] : [];

		$outVariants = [];
		foreach (self::page_transition_registry() as $variantKey => $_variantDef) {
			$variantKey     = (string) $variantKey;
			$variantSrc     = is_array($ptVariantsSrc[$variantKey] ?? null) ? $ptVariantsSrc[$variantKey] : [];
			$variantDefault = is_array($ptVariantsDefaults[$variantKey] ?? null) ? $ptVariantsDefaults[$variantKey] : [];

			$outVariants[$variantKey] = self::normalize_pt_variant($variantKey, $variantSrc, $variantDefault);
		}

		$out['pageTransition'] = [
			'enabled'       => $enabled,
			'activeVariant' => $activeVariant,
			'variants'      => $outVariants,
		];

		// General
		$generalSrc = $src['general'] ?? [];
		$generalSrc = is_array($generalSrc) ? $generalSrc : [];

		$remove = isset($generalSrc['removeDataOnUninstall'])
			? ((string) $generalSrc['removeDataOnUninstall'] === '1')
			: (bool) ($defaults['general']['removeDataOnUninstall'] ?? false);

		$out['general']['removeDataOnUninstall'] = $remove;

		return $out;
	}

	public static function sanitize_settings(array $src): array {
		return self::normalize_settings($src);
	}
}
