<?php

declare (strict_types = 1);

if (! defined('ABSPATH')) {
	exit;
}

final class WP_InView_Plugin {

	public static function init(): void {
		// Priorytet 1 — przed wszystkimi innymi hookami pluginu (prio 5).
		// Musi być na plugins_loaded, nie na init — inaczej __() wywołane
		// wcześniej (w register_setting, get_tabs, ui_schema) nie mają tłumaczeń.
		add_action('plugins_loaded', [self::class, 'load_textdomain'], 1);

		add_action('plugins_loaded', [self::class, 'on_plugins_loaded'], 5);
		add_action('init', [self::class, 'on_init'], 20);
	}

	/**
	 * Ładuje tłumaczenia pluginu z obsługą locale override z ustawień UI.
	 *
	 * Tryby:
	 *  'auto' → load_plugin_textdomain() — WP dobiera locale automatycznie.
	 *  'pl'   → wymusza wp-inview-pl_PL.mo niezależnie od locale WP.
	 *  'en'   → wymusza angielski (język bazowy kodu) przez unload istniejącego tłumaczenia.
	 *
	 * Używamy get_option() bezpośrednio — Settings::register() jeszcze się nie wykonało.
	 */
	public static function load_textdomain(): void {
		$raw  = get_option('wp_inview_settings', []);
		$lang = is_array($raw) ? (string) ($raw['ui']['language'] ?? 'auto') : 'auto';

		if ($lang === 'auto' || $lang === '') {
			load_plugin_textdomain(
				'wp-inview',
				false,
				dirname(plugin_basename(WP_INVIEW_FILE)) . '/languages'
			);
			return;
		}

		$locale_map = [
			'en' => 'en_US',
			'pl' => 'pl_PL',
		];

		$locale = $locale_map[$lang] ?? null;

		if ($locale === null) {
			load_plugin_textdomain(
				'wp-inview',
				false,
				dirname(plugin_basename(WP_INVIEW_FILE)) . '/languages'
			);
			return;
		}

		// Angielski = język bazowy — brak pliku .mo = oryginalne stringi z kodu.
		// Jeśli WP ma inny locale i załadował już tłumaczenia, wyładowujemy je.
		if ($locale === 'en_US') {
			if (is_textdomain_loaded('wp-inview')) {
				unload_textdomain('wp-inview');
			}
			return;
		}

		$mofile = WP_INVIEW_DIR . 'languages/wp-inview-' . $locale . '.mo';

		if (file_exists($mofile)) {
			if (is_textdomain_loaded('wp-inview')) {
				unload_textdomain('wp-inview');
			}
			load_textdomain('wp-inview', $mofile);
		} else {
			if (class_exists('WP_InView_Debug')) {
				WP_InView_Debug::warn('Missing .mo file for selected UI language', [
					'lang'   => $lang,
					'locale' => $locale,
					'path'   => $mofile,
				]);
			}
			load_plugin_textdomain(
				'wp-inview',
				false,
				dirname(plugin_basename(WP_INVIEW_FILE)) . '/languages'
			);
		}
	}

	public static function on_plugins_loaded(): void {
		if (class_exists('WP_InView_Settings')) {
			WP_InView_Settings::register();
		}

		if (class_exists('WP_InView_Assets')) {
			WP_InView_Assets::register();
		}

		if (is_admin() && class_exists('WP_InView_Admin')) {
			WP_InView_Admin::register();
		}
	}

	public static function on_init(): void {
		if (class_exists('WP_InView_ViewHelpers')) {
			WP_InView_ViewHelpers::register();
		}
	}
}