<?php

declare (strict_types = 1);

if (! defined('ABSPATH')) {
	exit;
}

final class WP_InView_Debug {

	private const PREFIX = '[WP InView]';

	/**
	 * Debug is enabled when either WP_INVIEW_DEBUG or WP_DEBUG is true.
	 */
	public static function enabled(): bool {
		return (defined('WP_INVIEW_DEBUG') && WP_INVIEW_DEBUG)
			|| (defined('WP_DEBUG') && WP_DEBUG);
	}

	/**
	 * Log a message with optional context array.
	 *
	 * @param array<string,mixed> $context
	 */
	public static function log(string $message, array $context = []): void {
		if (! self::enabled()) {
			return;
		}

		error_log(self::format('LOG', $message, $context));
	}

	/**
	 * @param array<string,mixed> $context
	 */
	public static function warn(string $message, array $context = []): void {
		if (! self::enabled()) {
			return;
		}

		error_log(self::format('WARN', $message, $context));
	}

	/**
	 * @param array<string,mixed> $context
	 */
	public static function error(string $message, array $context = []): void {
		if (! self::enabled()) {
			return;
		}

		error_log(self::format('ERROR', $message, $context));
	}

	/**
	 * Verify file exists (only in development).
	 * In production, assumes files exist (performance).
	 *
	 * @param string $path File path
	 * @param string $description File description for error message
	 * @return bool True if file exists or not checking
	 */
	public static function verify_file(string $path, string $description = 'file'): bool {
		if (! self::is_development()) {
			return true;
		}

		if (! file_exists($path)) {
			self::warn("Missing $description", [
				'path'   => $path,
				'exists' => false,
			]);
			return false;
		}

		return true;
	}

	/**
	 * Check if we're in development environment.
	 * This is separate from debug logging.
	 */
	public static function is_development(): bool {
		// 1) Explicit plugin dev flag
		if (defined('WP_INVIEW_DEBUG') && WP_INVIEW_DEBUG === true) {
			return true;
		}

		// 2) WP_DEBUG + dev-ish host
		if (! (defined('WP_DEBUG') && WP_DEBUG === true)) {
			return false;
		}

		$host = '';
		if (isset($_SERVER['HTTP_HOST'])) {
			$host = sanitize_text_field(wp_unslash($_SERVER['HTTP_HOST']));
		}

		// Normalize: lowercase + strip port (example.com:8080)
		$host = strtolower(trim($host));
		if ($host !== '' && strpos($host, ':') !== false) {
			$host = (string) strtok($host, ':');
		}

		if ($host === '') {
			return false;
		}

		$dev_patterns = [
			'localhost',
			'127.0.0.1',
			'.local',
			'.dev',
			'.test',
			'staging.',
		];

		foreach ($dev_patterns as $pattern) {
			if (stripos($host, $pattern) !== false) {
				return true;
			}
		}

		return false;
	}

	/**
	 * @param array<string,mixed> $context
	 */
	private static function format(string $level, string $message, array $context): string {
		$msg = trim($message);

		if ($context) {
			$json = wp_json_encode($context, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
			if (is_string($json) && $json !== '') {
				$msg .= ' | ' . $json;
			}
		}

		return self::PREFIX . ' ' . $level . ' ' . $msg;
	}
}