<?php

declare (strict_types = 1);

if (! defined('ABSPATH')) {
	exit;
}

final class WP_InView_Assets {

	/** @var array<string,mixed>|null */
	private static $globals_cache = null;

	/** Must match runtime key. */
	private const PAGE_TRANSITION_STORAGE_KEY = 'wp_inview_page_transition_v1';

	/** Must match runtime overlay id. */
	private const PAGE_TRANSITION_OVERLAY_ID = 'wp-inview-pt-overlay';

	/**
	 * Mapowanie ui.language → locale → nazwa pliku JSON z tłumaczeniami JS.
	 * Plik JSON musi być w formacie Jed 1.x (locale_data.messages).
	 */
	private const JS_TRANSLATION_FILES = [
		'pl' => 'wp-inview-pl_PL-index.json',
		// 'de' => 'wp-inview-de_DE-index.json',  // <- tutaj dokładasz kolejne języki
	];

	/**
	 * Whether any animated block has been rendered on this request.
	 * Set by ViewHelpers::inject_animation() via mark_frontend_needed().
	 */
	private static bool $frontend_needed = false;

	public static function register(): void {
		add_action('enqueue_block_editor_assets', [self::class, 'enqueue_editor']);

		// Register assets early (no HTTP request yet), enqueue only when needed.
		add_action('wp_enqueue_scripts', [self::class, 'register_frontend'], 1);

		// After all render_block filters have run, enqueue if anything was marked.
		add_action('wp_footer', [self::class, 'enqueue_frontend_if_needed'], 1);

		add_action('wp_head', [self::class, 'output_page_transition_bootstrap'], 0);
		add_action('wp_body_open', [self::class, 'output_page_transition_bootstrap'], 0);
	}

	/**
	 * Called by ViewHelpers::inject_animation() the first time it injects attrs
	 * into a block. Causes assets to be enqueued in wp_footer.
	 */
	public static function mark_frontend_needed(): void {
		self::$frontend_needed = true;
	}

	private static function read_inline_asset(string $path, string $type): string {
		if (! file_exists($path)) {
			WP_InView_Debug::warn("Missing inline {$type}", ['path' => $path]);
			return '';
		}

		$contents = file_get_contents($path);
		if (! is_string($contents) || $contents === '') {
			WP_InView_Debug::warn("Empty inline {$type}", ['path' => $path]);
			return '';
		}

		if ($type === 'css') {
			if (stripos($contents, '</style') !== false) {
				WP_InView_Debug::warn('Inline CSS contains </style>', ['path' => $path]);
				return '';
			}
			return $contents;
		}

		if ($type === 'js') {
			if (stripos($contents, '</script') !== false) {
				WP_InView_Debug::warn('Inline JS contains </script>', ['path' => $path]);
				return '';
			}
			$contents = preg_replace('/^\xEF\xBB\xBF/', '', $contents);
			return $contents;
		}

		return '';
	}

	public static function output_page_transition_bootstrap(): void {
		static $printed = false;
		if ($printed) {return;}
		$printed = true;

		$settings = class_exists('WP_InView_Settings') && method_exists('WP_InView_Settings', 'get')
			? (array) WP_InView_Settings::get()
			: [];

		$page_transition = isset($settings['pageTransition']) && is_array($settings['pageTransition']) ? $settings['pageTransition'] : [];
		$enabled         = (bool) ($page_transition['enabled'] ?? false);
		if (! $enabled) {return;}

		$css = self::read_inline_asset(WP_INVIEW_DIR . 'assets/css/bootstrap/__page-transition-bootstrap.css', 'css');
		if ($css !== '') {
			echo '<style id="wp-inview-pt-pre">' . $css . '</style>' . "\n";
		}

		$config = [
			'storageKey' => self::PAGE_TRANSITION_STORAGE_KEY,
			'className'  => 'wp-inview-pt-pre',
			'cssVarBg'   => '--wp-inview-pt-pre-bg',
			'fallbackMs' => 4000,
			'defaultBg'  => '#ffffff',
		];

		echo '<script>window.WP_INVIEW_PT_BOOTSTRAP_CONFIG=' . wp_json_encode($config) . ';</script>' . "\n";

		$js = self::read_inline_asset(WP_INVIEW_DIR . 'assets/js/bootstrap/__page-transition-bootstrap.js', 'js');
		if ($js !== '') {
			echo '<script>' . $js . '</script>' . "\n";
		}
	}

	public static function enqueue_editor(): void {
		$asset_file = WP_INVIEW_DIR . 'build/index.asset.php';

		if (! WP_InView_Debug::verify_file($asset_file, 'build asset manifest')) {
			if (WP_InView_Debug::is_development()) {
				return;
			}
		}

		$asset = require $asset_file;

		$deps    = isset($asset['dependencies']) && is_array($asset['dependencies']) ? $asset['dependencies'] : [];
		$version = isset($asset['version']) ? (string) $asset['version'] : (string) WP_INVIEW_VERSION;

		$index_js_path = WP_INVIEW_DIR . 'build/index.js';

		if (! WP_InView_Debug::verify_file($index_js_path, 'build asset manifest')) {
			if (WP_InView_Debug::is_development()) {
				return;
			}
		}

		wp_enqueue_script(
			'wp-inview-editor',
			trailingslashit(WP_INVIEW_URL) . 'build/index.js',
			$deps,
			$version,
			true
		);

		self::inject_js_translations('wp-inview-editor', 'wp-inview');

		$editor_css_path = WP_INVIEW_DIR . 'build/editor.css';
		WP_InView_Debug::verify_file($editor_css_path, 'editor stylesheet');

		if (file_exists($editor_css_path)) {
			$mtime   = filemtime($editor_css_path);
			$css_ver = $mtime !== false ? (string) $mtime : $version;

			wp_enqueue_style(
				'wp-inview-editor',
				trailingslashit(WP_INVIEW_URL) . 'build/editor.css',
				[],
				$css_ver
			);
		} else {
			WP_InView_Debug::warn('Missing editor stylesheet', ['path' => $editor_css_path]);
		}

		wp_add_inline_script(
			'wp-inview-editor',
			'window.WP_INVIEW_GLOBALS=' . wp_json_encode(self::get_globals()) . ';',
			'before'
		);
	}

	/**

	 * @param string $script_handle
	 * @param string $domain
	 */
	private static function inject_js_translations(string $script_handle, string $domain): void {
		$lang_file = self::resolve_js_translation_file();

		if ($lang_file === null) {

			return;
		}

		if (! file_exists($lang_file)) {
			WP_InView_Debug::warn('JS translation file not found', ['path' => $lang_file]);
			return;
		}

		$json = file_get_contents($lang_file);
		if (! is_string($json) || $json === '') {
			WP_InView_Debug::warn('JS translation file is empty', ['path' => $lang_file]);
			return;
		}

		$data = json_decode($json, true);
		if (! is_array($data) || ! isset($data['locale_data'])) {
			WP_InView_Debug::warn('Invalid JS translation file structure', ['path' => $lang_file]);
			return;
		}

		$script = sprintf(
			'( function( domain, translations ) {
				var localeData = translations.locale_data[ domain ] || translations.locale_data.messages;
				localeData[ "" ].domain = domain;
				wp.i18n.setLocaleData( localeData, domain );
			} )( %s, %s );',
			wp_json_encode($domain),
			$json
		);

		wp_add_inline_script($script_handle, $script, 'before');
	}

	/**
	 *
	 * @return string|null
	 */
	private static function resolve_js_translation_file(): ?string {
		$raw  = get_option('wp_inview_settings', []);
		$lang = is_array($raw) ? (string) ($raw['ui']['language'] ?? 'auto') : 'auto';

		if ($lang === 'en') {
			return null;
		}

		if ($lang !== 'auto' && isset(self::JS_TRANSLATION_FILES[$lang])) {
			return WP_INVIEW_DIR . 'languages/' . self::JS_TRANSLATION_FILES[$lang];
		}

		if ($lang === 'auto') {
			$locale = get_locale();

			if (strpos($locale, 'en_') === 0) {
				return null;
			}

			$locale_to_lang = [
				'pl_PL' => 'pl',
			];

			$mapped_lang = $locale_to_lang[$locale] ?? null;
			if ($mapped_lang !== null && isset(self::JS_TRANSLATION_FILES[$mapped_lang])) {
				return WP_INVIEW_DIR . 'languages/' . self::JS_TRANSLATION_FILES[$mapped_lang];
			}
		}

		return null;
	}

	/**
	 * Registers (but does NOT enqueue) frontend assets.
	 * Runs at priority 1 on wp_enqueue_scripts so assets are ready for
	 * wp_enqueue_style/script calls later in wp_footer.
	 *
	 * Also marks frontend as needed when Page Transition is active, because
	 * PT styles live in style.css and must load even on pages without animated
	 * blocks.
	 */
	public static function register_frontend(): void {
		// Page Transition styles are bundled in style.css. If PT is enabled,
		// mark frontend needed now — before any block has rendered.
		$settings   = class_exists('WP_InView_Settings') ? (array) WP_InView_Settings::get() : [];
		$pt         = isset($settings['pageTransition']) && is_array($settings['pageTransition']) ? $settings['pageTransition'] : [];
		$pt_enabled = (bool) ($pt['enabled'] ?? false);

		if ($pt_enabled) {
			self::$frontend_needed = true;
		}

		$style_path = WP_INVIEW_DIR . 'build/style.css';
		if (file_exists($style_path)) {
			$mtime   = filemtime($style_path);
			$css_ver = $mtime !== false ? (string) $mtime : (string) WP_INVIEW_VERSION;

			wp_register_style(
				'wp-inview',
				trailingslashit(WP_INVIEW_URL) . 'build/style.css',
				[],
				$css_ver
			);
		} else {
			WP_InView_Debug::warn('Missing frontend stylesheet', ['path' => $style_path]);
		}

		$runtime_path = WP_INVIEW_DIR . 'build/runtime.js';
		if (file_exists($runtime_path)) {
			$mtime  = filemtime($runtime_path);
			$js_ver = $mtime !== false ? (string) $mtime : (string) WP_INVIEW_VERSION;

			wp_register_script(
				'wp-inview-runtime',
				trailingslashit(WP_INVIEW_URL) . 'build/runtime.js',
				[],
				$js_ver,
				true
			);
		} else {
			WP_InView_Debug::warn('Missing runtime script', ['path' => $runtime_path]);
		}
	}

	/**
	 * Enqueues frontend assets only when at least one animated block was
	 * rendered. Hooked on wp_footer priority 1, after all render_block calls.
	 */
	public static function enqueue_frontend_if_needed(): void {
		if (! self::$frontend_needed) {
			return;
		}

		wp_enqueue_style('wp-inview');

		if (wp_script_is('wp-inview-runtime', 'registered')) {
			wp_enqueue_script('wp-inview-runtime');

			// Globals must be added after enqueue (before output).
			wp_add_inline_script(
				'wp-inview-runtime',
				'window.WP_INVIEW_GLOBALS=' . wp_json_encode(self::get_globals()) . ';',
				'before'
			);
		}
	}

	/**
	 * @return array<string,mixed>
	 */
	private static function get_globals(): array {
		$settings = [];
		if (class_exists('WP_InView_Settings') && method_exists('WP_InView_Settings', 'get')) {
			$settings = (array) WP_InView_Settings::get();
		}

		$raw = get_option('wp_inview_settings', []);

		$animation       = isset($settings['animation']) && is_array($settings['animation']) ? $settings['animation'] : [];
		$page_transition = isset($settings['pageTransition']) && is_array($settings['pageTransition']) ? $settings['pageTransition'] : [];

		$schema_version = 1;
		if (class_exists('WP_InView_Schema') && defined('WP_InView_Schema::SCHEMA_VERSION')) {
			$schema_version = (int) WP_InView_Schema::SCHEMA_VERSION;
		}

		return [
			'schemaVersion'  => $schema_version,
			'pluginVersion'  => defined('WP_INVIEW_VERSION') ? (string) WP_INVIEW_VERSION : '',
			'settingsUrl'    => admin_url('options-general.php?page=wp-inview'),
			'animation'      => $animation,
			'pageTransition' => $page_transition,
			'ui'             => array_merge(
				(array) WP_InView_Schema::ui_schema(),
				[
					'language'           => is_array($raw) ? (string) ($raw['ui']['language'] ?? 'auto') : 'auto',
					'editorPanelMode'    => is_array($raw) ? (string) ($raw['ui']['editorPanelMode'] ?? 'popup') : 'popup',
					'showListViewBadges' => is_array($raw) ? (bool) ($raw['ui']['showListViewBadges'] ?? true) : true,
				]
			),
			'defaults'       => (array) WP_InView_Schema::defaults(),
		];
	}
}