<?php

declare (strict_types = 1);

if (! defined('ABSPATH')) {
	exit;
}

final class WP_InView_Admin_ViewHelpers {
	private function __construct() {}

/**
 * @param array $field UI schema field definition
 * @param bool $extended Use extendedMax if available
 * @return string Safe HTML attributes string
 */
	public static function num_attrs(array $field, bool $extended = false): string {

		$min = null;
		if (array_key_exists('min', $field)) {
			$val = $field['min'];
			if (is_numeric($val)) {
				$min = (float) $val;
			} else {
				WP_InView_Debug::warn('Invalid min value in UI schema', [
					'field' => $field,
					'min'   => $val,
				]);
			}
		}

		// Extract and validate max
		$max = null;
		if ($extended && array_key_exists('extendedMax', $field)) {
			$val = $field['extendedMax'];
			if (is_numeric($val)) {
				$max = (float) $val;
			}
		} elseif (array_key_exists('max', $field)) {
			$val = $field['max'];
			if (is_numeric($val)) {
				$max = (float) $val;
			}
		}

		// Validate min <= max
		if ($min !== null && $max !== null && $min > $max) {
			WP_InView_Debug::warn('Invalid range: min > max', [
				'min' => $min,
				'max' => $max,
			]);
			// Swap them
			[$min, $max] = [$max, $min];
		}

		// Extract and validate step
		$step = null;
		if (array_key_exists('step', $field)) {
			$val = $field['step'];
			if (is_numeric($val) && $val > 0) {
				$step = (float) $val;
			}
		}

		// Build attributes string
		$out = '';

		if ($min !== null) {
			$out .= ' min="' . esc_attr((string) $min) . '"';
		}

		if ($max !== null) {
			$out .= ' max="' . esc_attr((string) $max) . '"';
		}

		if ($step !== null) {
			$out .= ' step="' . esc_attr((string) $step) . '"';
		}

		return $out;
	}

/**
 * @param array $field UI schema field definition
 * @return string Safe HTML attributes string
 */public static function label_help(string $label, string $help = ''): void {
		echo '<div class="wp-inview-field__label">' . esc_html($label) . '</div>';
		if ($help !== '') {
			echo '<p class="wp-inview-field__help">' . esc_html($help) . '</p>';
		}
	}
	public static function text_attrs(array $field): string {
		$maxLen = 0;

		if (array_key_exists('maxLen', $field)) {
			$val = $field['maxLen'];
			if (is_numeric($val) && $val > 0) {
				$maxLen = (int) $val;
			} else {
				WP_InView_Debug::warn('Invalid maxLen in UI schema', [
					'field'  => $field,
					'maxLen' => $val,
				]);
			}
		}

		return $maxLen > 0
			? ' maxlength="' . esc_attr((string) $maxLen) . '"'
			: '';
	}
	/**
	 * Hidden input used by chips.js to store the selected value.
	 */
	public static function hidden_value_input(string $name, string $value): void {
		printf(
			'<input type="hidden" data-wp-inview-value="1" name="%s" value="%s" />',
			esc_attr($name),
			esc_attr($value)
		);
	}

	/**
	 * Render chips list for a group.
	 *
	 * $chips item format: ['label' => '...', 'value' => '...'].
	 * $attr defaults to "data-value". For trigger presets etc. you might use "data-key".
	 *
	 * @param array<int,mixed> $chips
	 */
	public static function chips_list(array $chips, string $attr = 'data-value', string $active = ''): void {
		if (! $chips) {
			return;
		}

		$attr        = self::sanitize_attr_name($attr, 'data-value');
		$had_invalid = false;

		echo '<div class="wp-inview-chips">';

		foreach ($chips as $chip) {
			if (! is_array($chip)) {
				$had_invalid = true;
				continue;
			}

			$val = (string) ($chip['value'] ?? '');
			$lbl = (string) ($chip['label'] ?? '');

			if ($val === '' || $lbl === '') {
				$had_invalid = true;
				continue;
			}

			$classes = 'wp-inview-chip';
			if ($active !== '' && $active === $val) {
				$classes .= ' is-active';
			}

			printf(
				'<div class="%s" %s="%s"><span class="wp-inview-chip__dot" aria-hidden="true"></span><span class="wp-inview-chip__label">%s</span></div>',
				esc_attr($classes),
				esc_attr($attr),
				esc_attr($val),
				esc_html($lbl)
			);
		}

		echo '</div>';

		if ($had_invalid && class_exists('WP_InView_Debug')) {
			WP_InView_Debug::warn('Chips list contained invalid items', [
				'attr'   => $attr,
				'active' => $active,
			]);
		}
	}

	private static function sanitize_attr_name(string $attr, string $fallback): string {
		$a = trim($attr);

		// Allow only safe attribute name characters: letters, digits, colon, underscore, hyphen, dot.
		if ($a === '' || preg_match('/^[a-zA-Z_:][a-zA-Z0-9:._-]*$/', $a) !== 1) {
			if (class_exists('WP_InView_Debug')) {
				WP_InView_Debug::warn('Invalid attribute name for chips_list, using fallback', [
					'attr'     => $attr,
					'fallback' => $fallback,
				]);
			}
			return $fallback;
		}

		return $a;
	}
}