<?php

declare (strict_types = 1);

if (! defined('ABSPATH')) {
	exit;
}

final class WP_InView_Admin_Tab_Settings {

	private const TAB_KEY    = 'settings';
	private const PAGE_SLUG  = 'wp-inview';
	private const OPTION_KEY = 'wp_inview_settings';

	private const ACTION_RESET       = 'wp_inview_reset_settings';
	private const NONCE_ACTION_RESET = 'wp_inview_reset_settings';

	public static function key(): string {
		return self::TAB_KEY;
	}

	public static function label(): string {
		return __('Settings', 'wp-inview');
	}

	public static function register(): void {
		add_action('admin_post_' . self::ACTION_RESET, [self::class, 'handle_reset']);
	}

	private static function notice_missing_view(): void {
		echo '<div class="notice notice-error"><p>';
		echo esc_html__('Settings tab view file is missing. Check plugin deployment.', 'wp-inview');
		echo '</p></div>';
	}

	public static function render(): void {
		$raw            = get_option(self::OPTION_KEY, []);
		$lang           = is_array($raw) ? ($raw['ui']['language'] ?? 'auto') : 'auto';
		$panel_mode     = is_array($raw) ? ($raw['ui']['editorPanelMode'] ?? 'popup') : 'popup';
		$show_lv_badges = is_array($raw) ? (bool) ($raw['ui']['showListViewBadges'] ?? true) : true;

		// Flagi z query string po redirect.
		// phpcs:disable WordPress.Security.NonceVerification
		$just_reset = isset($_GET['reset']) && $_GET['reset'] === '1';
		$just_saved = isset($_GET['settings-updated']) && $_GET['settings-updated'] === 'true';
		// phpcs:enable

		$ctx = [
			'lang'           => $lang,
			'panel_mode'     => $panel_mode,
			'show_lv_badges' => $show_lv_badges,
			'just_reset'     => $just_reset,
			'just_saved'     => $just_saved,

			'option_key'     => self::OPTION_KEY,
			'page_slug'      => self::PAGE_SLUG,
			'tab_key'        => self::TAB_KEY,

			'view_base'      => WP_InView_Helpers::path('includes/Admin/Views/tabs/settings/'),
		];

		$view_path = WP_InView_Helpers::path('includes/Admin/Views/tabs/settings/layout.php');
		if (! file_exists($view_path)) {
			if (class_exists('WP_InView_Debug')) {
				WP_InView_Debug::error('Missing settings tab view file', ['path' => $view_path]);
			}
			self::notice_missing_view();
			return;
		}

		require $view_path;
	}

	public static function handle_reset(): void {
		if (! current_user_can('manage_options')) {
			wp_die(
				esc_html__('You do not have permission to reset settings.', 'wp-inview'),
				esc_html__('Forbidden', 'wp-inview'),
				['response' => 403]
			);
		}

		check_admin_referer(self::NONCE_ACTION_RESET);

		// Pełny reset przez Schema::defaults() — single source of truth.
		$defaults = method_exists('WP_InView_Schema', 'defaults')
			? (array) WP_InView_Schema::defaults()
			: ['ui' => ['language' => 'auto']];

		update_option(self::OPTION_KEY, $defaults);

		if (method_exists('WP_InView_Settings', 'flush_cache')) {
			WP_InView_Settings::flush_cache();
		}
		if (method_exists('WP_InView_Schema', 'flush_cache')) {
			WP_InView_Schema::flush_cache();
		}
		wp_safe_redirect(add_query_arg(
			[
				'page'  => self::PAGE_SLUG,
				'tab'   => self::TAB_KEY,
				'reset' => '1',
			],
			admin_url('options-general.php')
		));
		exit;
	}
}