<?php

declare (strict_types = 1);

if (! defined('ABSPATH')) {
	exit;
}

final class WP_InView_Admin_Tab_Animation {

	private static function safe_settings(): array {
		try {
			$s = WP_InView_Settings::get();
			return is_array($s) ? $s : [];
		} catch (\Throwable $e) {
			if (class_exists('WP_InView_Debug')) {
				WP_InView_Debug::error('Failed to load settings in Animation tab', [
					'message' => $e->getMessage(),
				]);
			}
			return [];
		}
	}

	private static function safe_schema_ui(): array {
		try {
			$ui = WP_InView_Schema::ui_schema();
			return is_array($ui) ? $ui : [];
		} catch (\Throwable $e) {
			if (class_exists('WP_InView_Debug')) {
				WP_InView_Debug::error('Failed to load schema UI in Animation tab', [
					'message' => $e->getMessage(),
				]);
			}
			return [];
		}
	}

	private static function arr($v): array {
		return is_array($v) ? $v : [];
	}

	private static function notice_missing_view(): void {
		echo '<div class="notice notice-error"><p>';
		echo esc_html(__('Animation tab view file is missing. Check plugin deployment.', 'wp-inview'));
		echo '</p></div>';
	}

	public static function render(): void {

		// ── Data ────────────────────────────────────────────────────────────
		$settings = self::safe_settings();
		$ui       = self::safe_schema_ui();

		$animation = self::arr($settings['animation'] ?? null);
		$presets   = self::arr($animation['presets'] ?? null);

		$registry = self::arr(WP_InView_Schema::presets_registry());
		$keys     = array_keys($registry);

		$defaults     = self::arr(WP_InView_Schema::defaults());
		$defaults_all = self::arr($defaults['animation']['presets'] ?? null);

		// ── UI schema (single extraction, passed down via $ctx) ──────────────
		$anim_ui   = self::arr($ui['animation'] ?? null);
		$ui_fields = self::arr($anim_ui['fields'] ?? null);
		$ui_copy   = self::arr($anim_ui['copy'] ?? null);
		$ui_info   = self::arr($anim_ui['info'] ?? null);

		$easing_rows  = self::arr($anim_ui['easing'] ?? null);
		$trigger_rows = self::arr($anim_ui['triggers'] ?? null);

		$opacity_chips = self::arr($ui_fields['opacityFrom']['chips'] ?? null);

		$preset_card_desc_tpl = (string) ($ui_copy['presetCardDescTpl'] ?? 'Applies to blocks using preset {preset}.');

		// ── ctx ─────────────────────────────────────────────────────────────
		$ctx = [
			// settings data
			'settings'             => $settings,
			'keys'                 => $keys,
			'presets'              => $presets,
			'defaults_all'         => $defaults_all,
			'registry'             => $registry,

			// ui schema — partials must NOT call Schema::* themselves
			'ui_fields'            => $ui_fields,
			'ui_copy'              => $ui_copy,
			'ui_info'              => $ui_info,
			'easing_rows'          => $easing_rows,
			'trigger_rows'         => $trigger_rows,
			'opacity_chips'        => $opacity_chips,
			'preset_card_desc_tpl' => $preset_card_desc_tpl,

			// path helper for partials
			'view_base'            => WP_InView_Helpers::path('includes/Admin/Views/tabs/animation/'),
		];

		$view_path = WP_InView_Helpers::path('includes/Admin/Views/tabs/animation/layout.php');
		if (! file_exists($view_path)) {
			if (class_exists('WP_InView_Debug')) {
				WP_InView_Debug::error('Missing animation tab view file', ['path' => $view_path]);
			}
			self::notice_missing_view();
			return;
		}

		require $view_path;
	}
}