<?php

declare (strict_types = 1);

if (! defined('ABSPATH')) {
	exit;
}

final class WP_InView_Admin_PageController {

	public static function render(string $menu_slug): void {
		// ============================================
		// HIGH PRIORITY FIX #6: Nonce in Tab Navigation
		// ============================================

		// Step 1: Verify user capability
		if (! current_user_can('manage_options')) {
			wp_die(
				esc_html__('You do not have sufficient permissions to access this page.', 'wp-inview'),
				esc_html__('Insufficient Permissions', 'wp-inview'),
				['response' => 403]
			);
		}

		$tabs = self::get_tabs();

		// Step 2: Get and sanitize tab parameter
		$tab_raw = isset($_GET['tab']) && is_string($_GET['tab'])
			? sanitize_key(wp_unslash($_GET['tab']))
			: 'animation';

		// Step 3: Verify tab exists
		if (! isset($tabs[$tab_raw])) {
			if ($tab_raw !== '' && class_exists('WP_InView_Debug')) {
				WP_InView_Debug::warn('Unknown admin tab, falling back to animation', [
					'tab' => $tab_raw,
				]);
			}
			$tab_raw = 'animation';
		} // Optional: Verify nonce if present (for future enhancement)
		if (isset($_GET['_wpnonce'])) {
			$nonce_action = 'wp_inview_tab_' . $tab_raw;
			if (! wp_verify_nonce(wp_unslash($_GET['_wpnonce']), $nonce_action)) {
				wp_die(
					esc_html__('Security check failed. Please try again.', 'wp-inview'),
					esc_html__('Security Error', 'wp-inview'),
					['response' => 403]
				);
			}
		}

		$current_tab = $tab_raw;

		// Step 4: Verify view file exists
		$view_path = WP_InView_Helpers::path('includes/Admin/Views/layout.php');
		if (! file_exists($view_path)) {
			if (class_exists('WP_InView_Debug')) {
				WP_InView_Debug::error('Missing admin page view file', ['path' => $view_path]);
			}

			wp_die(
				esc_html__('Plugin template file is missing. Please reinstall the plugin.', 'wp-inview'),
				esc_html__('Template Error', 'wp-inview'),
				['response' => 500]
			);
		}

		require $view_path;
	}

	/**
	 * @return array<string,array{label:string,render:callable}>
	 */
	private static function get_tabs(): array {
		$tab_labels = [];
		try {
			$ui         = WP_InView_Schema::ui_schema();
			$tab_labels = is_array( $ui['tabs'] ?? null ) ? $ui['tabs'] : [];
		} catch ( \Throwable $e ) {
			// Fallback to English labels if Schema is unavailable
		}

		return [
			'animation'  => [
				'label'  => $tab_labels['animation']  ?? __( 'Animation', 'wp-inview' ),
				'render' => [WP_InView_Admin_Tab_Animation::class, 'render'],
			],
			'cascade'    => [
				'label'  => $tab_labels['cascade']    ?? __( 'Cascade', 'wp-inview' ),
				'render' => [WP_InView_Admin_Tab_Cascade::class, 'render'],
			],
			'transition' => [
				'label'  => $tab_labels['transition'] ?? __( 'Page transition', 'wp-inview' ),
				'render' => [WP_InView_Admin_Tab_PageTransition::class, 'render'],
			],
			'settings'   => [
				'label'  => $tab_labels['settings']   ?? __( 'Settings', 'wp-inview' ),
				'render' => [WP_InView_Admin_Tab_Settings::class, 'render'],
			],
		];
	}
}