<?php

declare (strict_types = 1);

if (! defined('ABSPATH')) {
	exit;
}

final class WP_InView_Admin {

	private const MENU_SLUG = 'wp-inview';

	private static function get_current_tab(): string {
		if (! isset($_GET['tab'])) { // phpcs:ignore WordPress.Security.NonceVerification
			return 'animation';
		}

		$tab = sanitize_key(wp_unslash($_GET['tab'])); // phpcs:ignore WordPress.Security.NonceVerification

		// 'settings' musi być tutaj — bez tego get_current_tab() zwraca 'animation'
		// dla taba settings, co powoduje że enqueue_assets ładuje złe moduły JS.
		$valid_tabs = ['animation', 'cascade', 'transition', 'settings'];

		return in_array($tab, $valid_tabs, true) ? $tab : 'animation';
	}

	public static function register(): void {
		add_action('admin_menu', [self::class, 'register_page']);
		add_action('admin_enqueue_scripts', [self::class, 'enqueue_assets']);

		// Rejestracja handlera resetu z Settings taba.
		// Musi być tutaj — Admin::register() jest wywoływane z Plugin::on_plugins_loaded(),
		// więc hook admin_post_* jest zarejestrowany zanim WP przetworzy request.
		if (class_exists('WP_InView_Admin_Tab_Settings')) {
			WP_InView_Admin_Tab_Settings::register();
		}

		// Filtr redirect po zapisaniu przez options.php.
		// options.php zawsze przekierowuje na options.php?settings-updated=true
		// lub na wartość _wp_http_referer — bez taba settings.
		// Przechwytujemy i podmieniamy URL na właściwą stronę pluginu.
		add_filter('wp_redirect', [self::class, 'fix_save_redirect'], 10, 1);
	}

	/**
	 * Podmienia redirect po options.php na właściwą stronę pluginu z aktywnym tabem.
	 *
	 * Sprawdzamy $_POST['option_page'] === 'wp_inview' żeby nie psuć redirectów
	 * innych pluginów. Aktywny tab odczytujemy z ukrytego pola _wp_inview_current_tab
	 * dodanego przez layout.php do głównego formularza.
	 */
	public static function fix_save_redirect(string $location): string {
		// Działa tylko w kontekście POST do options.php.
		if (strpos($location, 'settings-updated') === false) {
			return $location;
		}

		// phpcs:ignore WordPress.Security.NonceVerification
		$option_page = isset($_POST['option_page']) ? sanitize_key($_POST['option_page']) : '';
		if ($option_page !== 'wp_inview') {
			return $location;
		}

		// Odczytaj aktywny tab z ukrytego pola formularza.
		// phpcs:ignore WordPress.Security.NonceVerification
		$tab = isset($_POST['_wp_inview_current_tab'])
			? sanitize_key($_POST['_wp_inview_current_tab'])
			: 'animation';

		$valid_tabs = ['animation', 'cascade', 'transition', 'settings'];
		if (! in_array($tab, $valid_tabs, true)) {
			$tab = 'animation';
		}

		return add_query_arg(
			[
				'page'             => self::MENU_SLUG,
				'tab'              => $tab,
				'settings-updated' => 'true',
			],
			admin_url('options-general.php')
		);
	}

	public static function register_page(): void {
		add_options_page(
			'WP InView',
			'WP InView',
			'manage_options',
			self::MENU_SLUG,
			[self::class, 'render_page']
		);
	}

	public static function enqueue_assets(string $hook): void {
		$page = isset($_GET['page']) ? sanitize_key((string) wp_unslash($_GET['page'])) : ''; // phpcs:ignore WordPress.Security.NonceVerification

		$is_plugin_page = ($hook === 'settings_page_' . self::MENU_SLUG)
			|| ($page === self::MENU_SLUG);

		if (! $is_plugin_page) {
			return;
		}

		$current_tab = self::get_current_tab();

		$admin_css_path = WP_INVIEW_DIR . 'assets/css/admin/admin.css';
		WP_InView_Debug::verify_file($admin_css_path, 'admin stylesheet');

		wp_enqueue_style(
			'wp-inview-admin',
			trailingslashit(WP_INVIEW_URL) . 'assets/css/admin/admin.css',
			[],
			WP_INVIEW_VERSION
		);

		$base_dir = WP_INVIEW_DIR . 'assets/js/admin/';
		$base_url = trailingslashit(WP_INVIEW_URL) . 'assets/js/admin/';

		$core_path = $base_dir . 'core.js';
		WP_InView_Debug::verify_file($core_path, 'admin script');

		wp_enqueue_script('wp-inview-admin-core', $base_url . 'core.js', [], WP_INVIEW_VERSION, true);

		$tab_modules = [
			'animation'  => [
				'dirty'    => ['file' => 'dirty.js', 'deps' => ['wp-inview-admin-core']],
				'chips'    => ['file' => 'chips.js', 'deps' => ['wp-inview-admin-core']],
				'duration' => ['file' => 'duration.js', 'deps' => ['wp-inview-admin-core']],
				'switches' => ['file' => 'switches.js', 'deps' => ['wp-inview-admin-core']],
				'preview'  => ['file' => 'preview.js', 'deps' => ['wp-inview-admin-core']],
			],
			'cascade'    => [
				'dirty'    => ['file' => 'dirty.js', 'deps' => ['wp-inview-admin-core']],
				'chips'    => ['file' => 'chips.js', 'deps' => ['wp-inview-admin-core']],
				'duration' => ['file' => 'duration.js', 'deps' => ['wp-inview-admin-core']],
			],
			'transition' => [
				'dirty'        => ['file' => 'dirty.js', 'deps' => ['wp-inview-admin-core']],
				'chips'        => ['file' => 'chips.js', 'deps' => ['wp-inview-admin-core']],
				'switches'     => ['file' => 'switches.js', 'deps' => ['wp-inview-admin-core']],
				'color-picker' => ['file' => 'color-picker.js', 'deps' => ['wp-inview-admin-core', 'wp-inview-admin-chips']],
			],
			// 'settings' tab ładuje tylko reset.js — lekki moduł bez zależności poza core.
			'settings'   => [
				'reset' => ['file' => 'reset.js', 'deps' => ['wp-inview-admin-core']],
			],
		];

		$modules = $tab_modules[$current_tab] ?? $tab_modules['animation'];

		foreach ($modules as $handle => $m) {
			$path = $base_dir . $m['file'];
			WP_InView_Debug::verify_file($path, 'admin script');

			wp_enqueue_script(
				'wp-inview-admin-' . $handle,
				$base_url . $m['file'],
				$m['deps'],
				WP_INVIEW_VERSION,
				true
			);
		}

		$index_deps = ['wp-inview-admin-core'];
		foreach (array_keys($modules) as $handle) {
			$index_deps[] = 'wp-inview-admin-' . $handle;
		}

		$index_path = $base_dir . 'index.js';
		WP_InView_Debug::verify_file($index_path, 'admin script');

		wp_enqueue_script(
			'wp-inview-admin-index',
			$base_url . 'index.js',
			$index_deps,
			WP_INVIEW_VERSION,
			true
		);

		$pkg = null;
		if (method_exists('WP_InView_Schema', 'package')) {
			$pkg = WP_InView_Schema::package();
		} else {
			WP_InView_Debug::warn('Schema::package not available, falling back to defaults+ui');
		}

		if (! is_array($pkg)) {
			$pkg = [
				'defaults' => method_exists('WP_InView_Schema', 'defaults') ? WP_InView_Schema::defaults() : [],
				'ui'       => method_exists('WP_InView_Schema', 'ui_schema') ? WP_InView_Schema::ui_schema() : [],
			];
		}

		wp_add_inline_script(
			'wp-inview-admin-index',
			'window.WP_INVIEW_GLOBALS = ' . wp_json_encode($pkg) . ';',
			'before'
		);
	}

	public static function render_page(): void {
		if (! current_user_can('manage_options')) {
			return;
		}

		WP_InView_Admin_PageController::render(self::MENU_SLUG);
	}
}