/* assets/js/admin/preview.js */
(function (w) {
	const Admin = w.WPInViewAdmin;
	if (!Admin) return;

	const Preview = {
		name: "preview",

		mount(root, ctx, api) {
			const cards = api.qsa('[data-wp-inview-card="1"]', root);
			if (!cards.length) return;

			const cleanups = [];

			cards.forEach((card) => {
				const off = initCardPreview(card, api);
				if (typeof off === "function") cleanups.push(off);
			});

			return () =>
				cleanups.forEach((fn) => {
					try {
						fn();
					} catch (e) {}
				});
		},
	};

	function initCardPreview(card, api) {
		const wrap = api.qs('[data-wp-inview-preview="1"]', card);
		if (!wrap) return null;

		const item = api.qs(".wp-inview-preview__item", wrap);
		if (!item) return null;

		const preset = card.getAttribute("data-wp-inview-preset") || "move";

		const qByName = (name) => card.querySelector(`[name="${CSS.escape(name)}"]`);

		const read = (key, fallback) => {
			const name = `wp_inview_settings[animation][presets][${preset}][${key}]`;
			const el = qByName(name);
			if (!el) return fallback;

			if (el.type === "checkbox") return el.checked ? "1" : "0";
			return el.value != null ? String(el.value) : fallback;
		};

		const clampInt = (v, min, max, def) => {
			const n = parseInt(String(v), 10);
			if (!Number.isFinite(n)) return def;
			return Math.min(max, Math.max(min, n));
		};

		const clampFloat = (v, min, max, def) => {
			const n = parseFloat(String(v));
			if (!Number.isFinite(n)) return def;
			return Math.min(max, Math.max(min, n));
		};

		const sanitizeEasing = (s) => {
			const v = String(s || "").trim();
			if (!v) return "cubic-bezier(0.22, 1, 0.36, 1)";
			if (!/^[a-zA-Z0-9\-\.\(\),\s]+$/.test(v)) {
				return "cubic-bezier(0.22, 1, 0.36, 1)";
			}
			if (v.length > 80) return v.slice(0, 80);
			return v;
		};

		const sanitizeEnum = (v, allowed, def) => {
			const s = String(v || "").trim();
			return allowed.includes(s) ? s : def;
		};

		const replay = () => {
			item.classList.remove("wp-inview--preview-in");
			void item.offsetHeight;
			item.classList.add("wp-inview--preview-in");
		};

		const applyCommon = () => {
			const duration = clampInt(read("duration", "650"), 0, 5000, 650);
			const delay = clampInt(read("delay", "0"), 0, 5000, 0);
			const opacityFrom = clampFloat(read("opacityFrom", "0"), 0, 1, 0);
			const easing = sanitizeEasing(read("easing", "cubic-bezier(0.22, 1, 0.36, 1)"));

			item.style.setProperty("--wp-inview-duration", `${duration}ms`);
			item.style.setProperty("--wp-inview-delay", `${delay}ms`);
			item.style.setProperty("--wp-inview-easing", easing);
			item.style.setProperty("--wp-inview-opacity-from", String(opacityFrom));
		};

		const applyMove = () => {
			const dist = clampInt(read("distance", "24"), 0, 300, 24);
			const dir = sanitizeEnum(read("direction", "up"), ["up", "down", "left", "right"], "up");

			let x = 0;
			let y = 0;

			if (dir === "up") y = dist;
			if (dir === "down") y = -dist;
			if (dir === "left") x = dist;
			if (dir === "right") x = -dist;

			item.style.setProperty("--wp-inview-x", `${x}px`);
			item.style.setProperty("--wp-inview-y", `${y}px`);
		};

		const applyFade = () => {
			item.style.setProperty("--wp-inview-x", "0px");
			item.style.setProperty("--wp-inview-y", "0px");
		};

		const applyZoom = () => {
			const zoomMode = sanitizeEnum(read("zoomMode", "in"), ["in", "out"], "in");
			const scaleFromRaw = clampFloat(read("scaleFrom", "0.95"), 0.2, 2.0, 0.95);

			const scaleFrom = zoomMode === "out" ? Math.max(0.2, Math.min(2.0, 1 + Math.abs(1 - scaleFromRaw))) : scaleFromRaw;

			const rubber = read("rubber", "0") === "1";

			item.style.setProperty("--wp-inview-scale-from", String(scaleFrom));
			item.setAttribute("data-wp-inview-preview-rubber", rubber ? "1" : "0");
		};

		const applyBlur = () => {
			const blurFrom = clampInt(read("blurFrom", "10"), 0, 40, 10);
			item.style.setProperty("--wp-inview-blur-from", `${blurFrom}px`);
		};

		const applyMask = () => {
			const variant = sanitizeEnum(read("maskVariant", "wipe"), ["wipe"], "wipe");
			const direction = sanitizeEnum(read("maskDirection", "up"), ["up", "down", "left", "right", "horizontal", "vertical"], "up");
			const feather = clampInt(read("maskFeather", "24"), 0, 100, 24);
			const inset = clampInt(read("maskInset", "20"), 0, 100, 20);

			item.setAttribute("data-wp-inview-mask-variant", variant);
			item.setAttribute("data-wp-inview-mask-direction", direction);

			item.style.setProperty("--wp-inview-mask-feather", `${feather}px`);
			item.style.setProperty("--wp-inview-mask-inset", `${inset}px`);
		};

		const update = () => {
			applyCommon();

			item.setAttribute("data-wp-inview-preview-preset", preset);

			if (preset === "move") applyMove();
			if (preset === "fade") applyFade();
			if (preset === "zoom") applyZoom();
			if (preset === "blur") applyBlur();
			if (preset === "mask") applyMask();

			replay();
		};

		const onInput = () => update();
		const onChange = () => update();

		card.addEventListener("input", onInput);
		card.addEventListener("change", onChange);

		const btn = api.qs('[data-wp-inview-preview-replay="1"]', card);
		let onBtn = null;

		if (btn) {
			onBtn = (e) => {
				e.preventDefault();
				replay();
			};
			btn.addEventListener("click", onBtn);
		}

		update();

		return () => {
			card.removeEventListener("input", onInput);
			card.removeEventListener("change", onChange);
			if (btn && onBtn) btn.removeEventListener("click", onBtn);
		};
	}

	Admin.register(Preview);
})(window);
