/* assets/js/admin/dirty.js */
(function (w, d) {
	const Admin = w.WPInViewAdmin;
	if (!Admin) return;

	const Dirty = {
		name: "dirty",
		init(root, ctx, api) {
			// Prevent double mounting
			if (Admin.__dirtyMounted) {
				console.warn("[WP InView] Dirty module already mounted");
				return;
			}
			Admin.__dirtyMounted = true;

			if (!Admin.state) Admin.state = {};
			if (typeof Admin.state.isDirty !== "boolean") Admin.state.isDirty = false;
			if (typeof Admin.state.isSaving !== "boolean") Admin.state.isSaving = false;

			// ============================================
			// CRITICAL FIX #4: Memory Leak Prevention
			// Track all event listeners for cleanup
			// ============================================
			const listeners = [];
			let isArmed = false;

			const markDirty = (ev) => {
				if (!isArmed) return;
				if (Admin.state.isSaving) return;
				if (ev && ev.isTrusted === false) return;
				Admin.state.isDirty = true;
			};

			const clearDirty = () => {
				Admin.state.isDirty = false;
			};

			const setSaving = () => {
				Admin.state.isSaving = true;
				clearDirty();
				w.setTimeout(() => {
					Admin.state.isSaving = false;
				}, 1500);
			};

			// Mark dirty on any input/select/textarea change
			api.qsa("input, select, textarea", root).forEach((el) => {
				const tag = (el.tagName || "").toLowerCase();
				const type = (el.getAttribute("type") || "").toLowerCase();

				if (tag === "input" && (type === "submit" || type === "button" || type === "reset")) {
					return;
				}

				// Add listeners and track them
				el.addEventListener("change", markDirty, { passive: true });
				el.addEventListener("input", markDirty, { passive: true });

				listeners.push({ el, type: "change", fn: markDirty });
				listeners.push({ el, type: "input", fn: markDirty });
			});

			// Save button click
			const saveBtn = api.qs("#submit", root) || api.qs('input[type="submit"][name="submit"]', root) || api.qs('button[type="submit"]', root);

			if (saveBtn) {
				const handleSaveClick = () => setSaving();
				saveBtn.addEventListener("click", handleSaveClick, { capture: true });
				listeners.push({ el: saveBtn, type: "click", fn: handleSaveClick, capture: true });
			}

			// Form submit
			api.qsa("form", root).forEach((form) => {
				const handleSubmit = () => setSaving();
				form.addEventListener("submit", handleSubmit, { capture: true });
				listeners.push({ el: form, type: "submit", fn: handleSubmit, capture: true });
			});

			// Beforeunload warning
			const handleBeforeUnload = (e) => {
				if (Admin.state.isSaving) return;
				if (!Admin.state.isDirty) return;
				e.preventDefault();
				e.returnValue = "";
			};

			w.addEventListener("beforeunload", handleBeforeUnload);
			listeners.push({ el: w, type: "beforeunload", fn: handleBeforeUnload });

			// Arm after initial mounts
			w.requestAnimationFrame(() => {
				w.requestAnimationFrame(() => {
					isArmed = true;
				});
			});

			// Expose helpers
			Admin.setDirty = () => {
				if (Admin.state.isSaving) return;
				Admin.state.isDirty = true;
			};
			Admin.clearDirty = () => clearDirty();
			Admin.isDirty = () => !!Admin.state.isDirty;
			Admin.setSaving = () => setSaving();

			return () => {
				// Remove all tracked event listeners
				listeners.forEach(({ el, type, fn, capture }) => {
					try {
						el.removeEventListener(type, fn, capture ? { capture: true } : undefined);
					} catch (e) {
						console.error("[WP InView] Failed to remove listener:", e);
					}
				});

				// Clear listeners array
				listeners.length = 0;

				// Reset mounted flag
				Admin.__dirtyMounted = false;

				// Clear state helpers
				delete Admin.setDirty;
				delete Admin.clearDirty;
				delete Admin.isDirty;
				delete Admin.setSaving;
			};
		},
	};

	Admin.register(Dirty);
})(window, document);
