/* assets/js/admin/core.js */
(function (w) {
	const Admin = {
		modules: [],
		_cleanups: [],
		_isMounted: false,

		state: {
			dirty: false,
		},

		isDirty() {
			return !!this.state.dirty;
		},

		setDirty(flag) {
			const next = !!flag;
			if (this.state.dirty === next) return;
			this.state.dirty = next;

			try {
				document.dispatchEvent(new CustomEvent("wp-inview-admin:dirty", { detail: { dirty: next } }));
			} catch (e) {}
		},

		clearDirty() {
			this.setDirty(false);
		},

		register(mod) {
			if (!mod) return;

			const hasInit = typeof mod.init === "function";
			const hasMount = typeof mod.mount === "function";

			if (!hasInit && !hasMount) return;

			this.modules.push(mod);
		},

		qs(sel, root) {
			return (root || document).querySelector(sel);
		},

		qsa(sel, root) {
			return Array.from((root || document).querySelectorAll(sel));
		},

		getRoot() {
			return document.querySelector('[data-wp-inview-admin="1"]');
		},

		getCtx() {
			const g = w.WP_INVIEW_GLOBALS || {};
			// Detect current tab
			const tab = this.getCurrentTab(); // animation, cascade, transition

			let ui = {};
			if (tab === "animation" || tab === "cascade") {
				ui = g.ui && g.ui.animation ? g.ui.animation : {};
			} else if (tab === "transition") {
				ui = g.ui && g.ui.pageTransition ? g.ui.pageTransition : {};
			}

			return { globals: g, ui };
		},

		getCurrentTab() {
			// Read from URL
			const params = new URLSearchParams(window.location.search);
			return params.get("tab") || "animation";
		},

		unmount() {
			if (!this._isMounted) return;

			for (let i = this._cleanups.length - 1; i >= 0; i--) {
				const item = this._cleanups[i];
				try {
					if (item && typeof item.fn === "function") item.fn();
				} catch (e) {
					console.error("[WP InView] Admin cleanup failed:", item && item.name ? item.name : "unknown", e);
				}
			}

			this._cleanups = [];
			this._isMounted = false;
		},

		boot() {
			const root = this.getRoot();
			if (!root) return;

			this.unmount();

			const ctx = this.getCtx();
			this._isMounted = true;

			this.modules.forEach((m) => {
				const name = m && m.name ? m.name : "unknown";

				try {
					const fn = typeof m.mount === "function" ? m.mount : m.init;

					const cleanup = fn.call(m, root, ctx, this);

					if (typeof cleanup === "function") {
						this._cleanups.push({ name, fn: cleanup });
					}
				} catch (e) {
					console.error("[WP InView] Admin module failed:", name, e);
				}
			});
		},

		reboot() {
			this.boot();
		},
	};

	w.WPInViewAdmin = Admin;
})(window);
