/* assets/js/admin/chips.js */
(function (w) {
	const Admin = w.WPInViewAdmin;
	if (!Admin) return;

	const Chips = {
		name: "chips",

		init(root, ctx, api) {
			const controller = typeof AbortController !== "undefined" ? new AbortController() : null;
			const off = [];

			const on = (el, type, fn, opts) => {
				if (!el) return;

				if (controller && controller.signal) {
					try {
						el.addEventListener(type, fn, Object.assign({ signal: controller.signal }, opts || {}));
						return;
					} catch (e) {}
				}

				el.addEventListener(type, fn, opts || false);
				off.push(() => {
					try {
						el.removeEventListener(type, fn, opts || false);
					} catch (e) {}
				});
			};

			// Guard against marking dirty during initialisation.
			let isReady = false;
			w.requestAnimationFrame(() => {
				w.requestAnimationFrame(() => {
					isReady = true;
				});
			});

			const markDirty = () => {
				if (!isReady) return;
				Admin.setDirty(true);
			};

			const groups = api.qsa("[data-wp-inview-chip-group]", root);
			if (!groups.length) {
				this.initScaleSync(root, api, ctx.ui, on, markDirty);
				return () => {
					if (controller) controller.abort();
					off.forEach((fn) => fn());
				};
			}

			const triggerMap = this.buildTriggerMap(ctx.ui);

			groups.forEach((group) => {
				const type = String(group.getAttribute("data-wp-inview-chip-group") || "").trim();
				if (!type) return;

				// Generic value groups (work for both tabs)
				if (type === "opacity") this.initValueGroup(group, api, on, markDirty, { seed: "0.2", prefillCustomFromLastPreset: true });
				if (type === "easing") this.initValueGroup(group, api, on, markDirty, { seed: "cubic-bezier(0.22, 1, 0.36, 1)", prefillCustomFromLastPreset: true });

				// Page transition specific (no custom input needed)
				if (type === "variant") this.initValueGroupNoCustom(group, api, on, markDirty, { seed: "fade" });
				if (type.startsWith("color-")) this.initValueGroup(group, api, on, markDirty, { seed: "#ffffff", prefillCustomFromLastPreset: true });
				if (type.startsWith("opacity-")) this.initValueGroup(group, api, on, markDirty, { seed: "1", prefillCustomFromLastPreset: true });
				if (type.startsWith("easing-")) this.initValueGroup(group, api, on, markDirty, { seed: "cubic-bezier(0.22, 1, 0.36, 1)", prefillCustomFromLastPreset: true });
				if (type === "curtain-direction") this.initValueGroupNoCustom(group, api, on, markDirty, { seed: "left" });

				// Animation-specific (require animation UI)
				if (ctx.ui && ctx.ui.triggers) {
					if (type === "trigger") this.initTriggerGroup(group, api, on, markDirty, triggerMap);
				}
				if (type === "rubber") this.initRubberGroup(group, api, on, markDirty);
				if (type === "maskDirection") this.initValueGroupNoCustom(group, api, on, markDirty, { seed: "right" });
				if (type === "direction") this.initValueGroupNoCustom(group, api, on, markDirty, { seed: "up" });
				if (type === "zoomMode") this.initZoomModeGroup(group, api, on, markDirty);
			});

			// Only init scaleSync if we have animation UI
			if (ctx.ui && ctx.ui.fields && ctx.ui.fields.scaleFrom) {
				this.initScaleSync(root, api, ctx.ui, on, markDirty);
			}

			return () => {
				if (controller) controller.abort();
				off.forEach((fn) => fn());
			};
		},

		buildTriggerMap(ui) {
			const rows = Array.isArray(ui.triggers) ? ui.triggers : [];
			const map = {};
			rows.forEach((r) => {
				if (!r) return;
				const type = String(r.type || "");
				const key = String(r.key || "");
				if (!key || type !== "preset") return;
				map[key] = { threshold: r.threshold, rootMargin: r.rootMargin };
			});
			return map;
		},

		setActive(group, value, api, attr) {
			api.qsa(".wp-inview-chip", group).forEach((chip) => {
				const v = String(chip.getAttribute(attr) || "");
				chip.classList.toggle("is-active", v === String(value));
			});
		},

		getFirstChipValue(group, api, attr) {
			const chip = api.qs(".wp-inview-chip[" + attr + "]", group);
			return chip ? String(chip.getAttribute(attr) || "").trim() : "";
		},

		initValueGroup(group, api, on, markDirty, opts) {
			const chips = api.qsa(".wp-inview-chip[data-value]", group);
			const hidden = api.qs('[data-wp-inview-value="1"]', group);
			const wrap = api.qs('[data-wp-inview-custom-wrap="1"]', group);
			const input = api.qs('[data-wp-inview-custom-input="1"]', group);

			if (!hidden || !chips.length) return;

			const seed = opts && opts.seed ? String(opts.seed) : this.getFirstChipValue(group, api, "data-value") || "";
			const isPresetValue = (val) => chips.some((c) => String(c.getAttribute("data-value")) === String(val) && String(val) !== "custom");

			const apply = (val) => {
				const v = String(val);

				this.setActive(group, v, api, "data-value");

				const isCustom = v === "custom";
				if (wrap) wrap.classList.toggle("wp-inview-hidden", !isCustom);

				if (!isCustom) {
					if (hidden.value !== v) {
						hidden.value = v;
						markDirty();
					}
					return;
				}

				const current = String(hidden.value || "").trim();

				if (!input) {
					if (!current && seed) {
						hidden.value = seed;
						markDirty();
					}
					return;
				}

				const prefill = !!(opts && opts.prefillCustomFromLastPreset);

				if (prefill) {
					input.value = isPresetValue(current) ? current : current || seed;
				} else {
					if (!String(input.value || "").trim()) {
						input.value = isPresetValue(current) ? seed : current || seed;
					}
				}

				const next = String(input.value || "").trim();
				if (next && hidden.value !== next) {
					hidden.value = next;
					markDirty();
				}
			};

			const current = String(hidden.value || "").trim();
			apply(isPresetValue(current) ? current : "custom");

			chips.forEach((chip) => {
				on(chip, "click", (e) => {
					e.preventDefault();
					const v = String(chip.getAttribute("data-value") || "");
					apply(v);
					if (v === "custom" && input) input.focus();
				});
			});

			if (input) {
				on(input, "input", () => {
					const v = String(input.value || "").trim();
					if (v && hidden.value !== v) {
						hidden.value = v;
						markDirty();
					}
				});
			}
		},

		initValueGroupNoCustom(group, api, on, markDirty, opts) {
			const chips = api.qsa(".wp-inview-chip[data-value]", group);
			const hidden = api.qs('[data-wp-inview-value="1"]', group);
			if (!hidden || !chips.length) return;

			const isAllowed = (val) => chips.some((c) => String(c.getAttribute("data-value") || "") === String(val));
			const first = this.getFirstChipValue(group, api, "data-value");
			const seed = (opts && opts.seed ? String(opts.seed) : "") || first;

			const apply = (val) => {
				const v = String(val || "").trim();
				const current = String(hidden.value || "").trim();

				let next = v && isAllowed(v) ? v : "";
				if (!next && current && isAllowed(current)) next = current;
				if (!next && seed && isAllowed(seed)) next = seed;
				if (!next && first && isAllowed(first)) next = first;

				if (!next) return;

				if (hidden.value !== next) {
					hidden.value = next;
					markDirty();
				}
				this.setActive(group, next, api, "data-value");
			};

			apply(String(hidden.value || "").trim());

			chips.forEach((chip) => {
				on(chip, "click", (e) => {
					e.preventDefault();
					apply(String(chip.getAttribute("data-value") || ""));
				});
			});
		},

		initZoomModeGroup(group, api, on, markDirty) {
			const chips = api.qsa(".wp-inview-chip[data-value]", group);
			const hidden = api.qs('[data-wp-inview-value="1"]', group);
			if (!hidden || !chips.length) return;

			const card = group.closest('[data-wp-inview-card="1"]');
			const modeSelect = card ? api.qs('[data-wp-inview-zoom-mode="1"]', card) : null;

			const isAllowed = (val) => chips.some((c) => String(c.getAttribute("data-value") || "") === String(val));
			const firstChip = this.getFirstChipValue(group, api, "data-value");

			const safeMode = (raw) => {
				const v = String(raw || "").trim();
				if (v && isAllowed(v)) return v;

				const current = String(hidden.value || "").trim();
				if (current && isAllowed(current)) return current;

				if (firstChip && isAllowed(firstChip)) return firstChip;
				return "in";
			};

			const dispatchChange = (el) => {
				if (!el) return;
				try {
					el.dispatchEvent(new Event("change", { bubbles: true }));
				} catch (e) {}
			};

			const syncSelect = (val) => {
				if (!modeSelect) return;

				const v = String(val || "").trim();
				const exists = modeSelect.querySelector('option[value="' + CSS.escape(v) + '"]');
				if (exists) {
					modeSelect.value = v;
				} else if (modeSelect.options && modeSelect.options.length) {
					modeSelect.value = String(modeSelect.options[0].value || "");
				}
				dispatchChange(modeSelect);
			};

			const apply = (val) => {
				const safe = safeMode(val);

				if (hidden.value !== safe) {
					hidden.value = safe;
					markDirty();
				}
				this.setActive(group, safe, api, "data-value");
				syncSelect(safe);
			};

			apply(String(hidden.value || "").trim());

			chips.forEach((chip) => {
				on(chip, "click", (e) => {
					e.preventDefault();
					apply(String(chip.getAttribute("data-value") || ""));
				});
			});
		},

		initRubberGroup(group, api, on, markDirty) {
			const chips = api.qsa(".wp-inview-chip[data-key]", group);
			const hiddenMode = api.qs('[data-wp-inview-value="1"]', group);
			const hiddenAmp = api.qs('[data-wp-inview-rubber-amp="1"]', group);
			const hiddenBool = api.qs('[data-wp-inview-rubber-bool="1"]', group);
			const wrap = api.qs('[data-wp-inview-custom-wrap="1"]', group);
			const input = api.qs('[data-wp-inview-rubber-amp-percent="1"]', group);

			if (!hiddenMode || !chips.length) return;

			const ampPresets = { off: 0, soft: 0.02, medium: 0.035, hard: 0.055 };
			const clamp = (n, min, max) => Math.max(min, Math.min(max, n));

			const setAmp = (amp) => {
				const a = Number(amp);
				if (!hiddenAmp) return;
				const next = String(clamp(isFinite(a) ? a : 0, 0, 0.1));
				if (hiddenAmp.value !== next) {
					hiddenAmp.value = next;
					markDirty();
				}
			};

			const setBool = (mode) => {
				if (!hiddenBool) return;
				const next = mode && mode !== "off" ? "1" : "0";
				if (hiddenBool.value !== next) {
					hiddenBool.value = next;
					markDirty();
				}
			};

			const apply = (mode) => {
				const m = String(mode || "off");

				if (hiddenMode.value !== m) {
					hiddenMode.value = m;
					markDirty();
				}

				this.setActive(group, m, api, "data-key");

				const isCustom = m === "custom";
				if (wrap) wrap.classList.toggle("wp-inview-hidden", !isCustom);

				setBool(m);

				if (m === "off") {
					setAmp(0);
					if (input) input.value = "0";
					return;
				}

				if (!isCustom) {
					const presetAmp = ampPresets[m] ?? 0.035;
					setAmp(presetAmp);
					if (input) input.value = String(Math.round(presetAmp * 1000) / 10);
					return;
				}

				if (input) {
					const current = Number(parseFloat(String(input.value || "").replace(",", ".")));
					const fallback = hiddenAmp ? Number(parseFloat(String(hiddenAmp.value || ""))) : 0.035;
					const amp = isFinite(current) ? current / 100 : isFinite(fallback) ? fallback : 0.035;
					input.value = String(Math.round(clamp(amp, 0, 0.1) * 1000) / 10);
					setAmp(amp);
				}
			};

			apply(String(hiddenMode.value || "").trim() || "off");

			chips.forEach((chip) => {
				on(chip, "click", (e) => {
					e.preventDefault();
					const k = String(chip.getAttribute("data-key") || "off");
					apply(k);
					if (k === "custom" && input) input.focus();
				});
			});

			if (input) {
				on(input, "input", () => {
					const pct = Number(parseFloat(String(input.value || "").replace(",", ".")));
					if (!isFinite(pct)) return;
					setAmp(clamp(pct / 100, 0, 0.1));
				});
			}
		},

		initScaleSync(root, api, ui, on, markDirty) {
			const wraps = api.qsa('[data-wp-inview-scale-wrap="1"]', root);
			if (!wraps.length) return;

			const clamp = (n, min, max) => Math.max(min, Math.min(max, n));

			wraps.forEach((wrap) => {
				const percentInput = api.qs('[data-wp-inview-scale-percent="1"]', wrap);
				const hiddenScale = api.qs('[data-wp-inview-scale-value="1"]', wrap);
				const readout = api.qs('[data-wp-inview-scale-readout="1"]', wrap);

				const card = wrap.closest('[data-wp-inview-card="1"]') || root;
				const modeSelect = api.qs('[data-wp-inview-zoom-mode="1"]', card);

				if (!percentInput || !hiddenScale) return;

				const getRanges = () => {
					const modes = ui && ui.fields && ui.fields.scaleFrom && ui.fields.scaleFrom.modes ? ui.fields.scaleFrom.modes : {};
					const mode = modeSelect ? String(modeSelect.value || "in") : "in";
					const r = modes[mode] || {};
					return {
						min: Number(r.min ?? (mode === "out" ? 1.01 : 0.85)),
						max: Number(r.max ?? (mode === "out" ? 1.2 : 0.99)),
					};
				};

				const syncFromPercent = () => {
					const ranges = getRanges();

					const minPct = Math.round(ranges.min * 100);
					const maxPct = Math.round(ranges.max * 100);
					try {
						percentInput.min = String(minPct);
						percentInput.max = String(maxPct);
					} catch (e) {}

					const raw = Number(String(percentInput.value || "100").replace(",", "."));
					const pct = isFinite(raw) ? raw : 100;

					const scale = clamp(pct / 100, ranges.min, ranges.max);

					const nextScale = String(scale);
					if (hiddenScale.value !== nextScale) {
						hiddenScale.value = nextScale;
						markDirty();
					}

					const nextPct = String(Math.round(scale * 100));
					if (String(percentInput.value || "") !== nextPct) {
						percentInput.value = nextPct;
						// tu nie markDirty, bo to jest korekta UI po clampie
					}

					if (readout) readout.textContent = scale.toFixed(2);
				};

				on(percentInput, "input", syncFromPercent);
				if (modeSelect) on(modeSelect, "change", syncFromPercent);

				syncFromPercent();
			});
		},

		initTriggerGroup(group, api, on, markDirty, triggerMap) {
			const chips = api.qsa(".wp-inview-chip[data-key]", group);
			const wrap = api.qs('[data-wp-inview-custom-wrap="1"]', group);
			const thr = api.qs('[data-wp-inview-trigger-threshold="1"]', group);
			const rm = api.qs('[data-wp-inview-trigger-rootMargin="1"]', group);

			const presetHidden = api.qs('[data-wp-inview-trigger-preset="1"]', group) || api.qs('[data-wp-inview-value="1"]', group);

			if (!chips.length || !thr || !rm || !presetHidden) return;

			const normalizeRM = (v) => String(v || "").trim();

			const matchPreset = () => {
				const threshold = Number(parseFloat(String(thr.value || "")));
				const rootMargin = normalizeRM(rm.value);

				let found = "custom";
				Object.keys(triggerMap || {}).forEach((k) => {
					const row = triggerMap[k];
					if (!row) return;

					const rowThr = Number(row.threshold);
					const rowRm = normalizeRM(row.rootMargin);

					if (rowThr === threshold && rowRm === rootMargin) {
						found = k;
					}
				});
				return found;
			};

			const apply = (key, opts) => {
				const k = String(key || "custom").trim() || "custom";
				const silent = !!(opts && opts.silent);

				if (presetHidden.value !== k) {
					presetHidden.value = k;
					if (!silent) markDirty();
				}

				this.setActive(group, k, api, "data-key");

				const isCustom = k === "custom";
				if (wrap) wrap.classList.toggle("wp-inview-hidden", !isCustom);

				if (isCustom) return;

				const row = triggerMap && triggerMap[k] ? triggerMap[k] : null;
				if (!row) return;

				if (String(thr.value) !== String(row.threshold)) {
					thr.value = String(row.threshold);
					if (!silent) markDirty();
				}
				if (String(rm.value) !== String(row.rootMargin)) {
					rm.value = String(row.rootMargin);
					if (!silent) markDirty();
				}
			};

			// init: prefer saved preset if valid, else derive from values
			const initialPreset = String(presetHidden.value || "").trim();
			if (initialPreset && initialPreset !== "custom" && triggerMap && triggerMap[initialPreset]) {
				apply(initialPreset, { silent: true });
			} else {
				apply(matchPreset(), { silent: true });
			}

			// clicks
			chips.forEach((chip) => {
				on(chip, "click", (e) => {
					e.preventDefault();
					apply(String(chip.getAttribute("data-key") || "custom"));
				});
			});

			// manual edits
			const onManual = () => {
				apply(matchPreset());
			};

			on(thr, "input", onManual);
			on(rm, "input", onManual);
		},
	};

	Admin.register(Chips);
})(window);
